<?php

namespace App\Exports;

use Maatwebsite\Excel\Concerns\FromArray;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithTitle;
use Maatwebsite\Excel\Concerns\WithStyles;
use Maatwebsite\Excel\Concerns\WithColumnWidths;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;

class ReportExport implements FromArray, WithHeadings, WithTitle, WithStyles, WithColumnWidths
{
    protected $reportData;
    protected $reportType;

    public function __construct(array $reportData, string $reportType)
    {
        $this->reportData = $reportData;
        $this->reportType = $reportType;
    }

    public function array(): array
    {
        $data = [];

        switch ($this->reportType) {
            case 'users':
                foreach ($this->reportData['data'] as $user) {
                    $data[] = [
                        $user->name ?? '',
                        $user->email ?? '',
                        $user->employee_id ?? '',
                        $user->department->name ?? 'N/A',
                        $user->role ?? '',
                        $user->status ?? '',
                        $user->phone ?? '',
                        $user->created_at->format('Y-m-d') ?? '',
                    ];
                }
                break;

            case 'visits':
                foreach ($this->reportData['data'] as $visit) {
                    $data[] = [
                        $visit->employee->name ?? '',
                        $visit->client->name ?? '',
                        $visit->client->company_name ?? '',
                        $visit->governorate->name ?? '',
                        $visit->city->name ?? '',
                        $visit->area ?? '',
                        $visit->scheduled_date->format('Y-m-d') ?? '',
                        $visit->scheduled_time ?? '',
                        $visit->status ?? '',
                        $visit->comments ?? '',
                    ];
                }
                break;

            case 'leaves':
                foreach ($this->reportData['data'] as $leave) {
                    $data[] = [
                        $leave->employee->name ?? '',
                        $leave->type ?? '',
                        $leave->start_date->format('Y-m-d') ?? '',
                        $leave->end_date->format('Y-m-d') ?? '',
                        $leave->total_days ?? '',
                        $leave->status ?? '',
                        $leave->reason ?? '',
                        $leave->approver->name ?? 'N/A',
                        $leave->created_at->format('Y-m-d') ?? '',
                    ];
                }
                break;

            case 'payroll':
                foreach ($this->reportData['data'] as $payroll) {
                    $data[] = [
                        $payroll->employee->name ?? '',
                        $payroll->pay_period_start->format('Y-m-d') ?? '',
                        $payroll->pay_period_end->format('Y-m-d') ?? '',
                        number_format($payroll->basic_salary, 2),
                        number_format($payroll->allowances, 2),
                        number_format($payroll->deductions, 2),
                        number_format($payroll->overtime, 2),
                        number_format($payroll->bonus, 2),
                        number_format($payroll->net_salary, 2),
                        $payroll->status ?? '',
                        $payroll->payment_date ? $payroll->payment_date->format('Y-m-d') : '',
                        $payroll->notes ?? '',
                    ];
                }
                break;

            case 'clients':
                foreach ($this->reportData['data'] as $client) {
                    $data[] = [
                        $client->name ?? '',
                        $client->company_name ?? '',
                        $client->governorate->name ?? '',
                        $client->city->name ?? '',
                        $client->area ?? '',
                        $client->address ?? '',
                        $client->phone ?? '',
                        $client->email ?? '',
                        $client->type ?? '',
                        $client->status ?? '',
                        $client->created_at->format('Y-m-d') ?? '',
                    ];
                }
                break;

            case 'performance':
                foreach ($this->reportData['data'] as $performance) {
                    $data[] = [
                        $performance['employee']['name'] ?? '',
                        $performance['department'] ?? '',
                        $performance['total_visits'] ?? 0,
                        $performance['completed_visits'] ?? 0,
                        number_format($performance['completion_rate'], 2) . '%',
                        $performance['total_leave_days'] ?? 0,
                        $performance['avg_visit_duration'] ?? 0,
                    ];
                }
                break;
        }

        return $data;
    }

    public function headings(): array
    {
        switch ($this->reportType) {
            case 'users':
                return ['Name', 'Email', 'Employee ID', 'Department', 'Role', 'Status', 'Phone', 'Join Date'];
            
            case 'visits':
                return ['Employee', 'Client Name', 'Company', 'Governorate', 'City', 'Area', 'Date', 'Time', 'Status', 'Comments'];
            
            case 'leaves':
                return ['Employee', 'Type', 'Start Date', 'End Date', 'Total Days', 'Status', 'Reason', 'Approver', 'Request Date'];
            
            case 'payroll':
                return ['Employee', 'Period Start', 'Period End', 'Basic Salary', 'Allowances', 'Deductions', 'Overtime', 'Bonus', 'Net Salary', 'Status', 'Payment Date', 'Notes'];
            
            case 'clients':
                return ['Client Name', 'Company', 'Governorate', 'City', 'Area', 'Address', 'Phone', 'Email', 'Type', 'Status', 'Created Date'];
            
            case 'performance':
                return ['Employee', 'Department', 'Total Visits', 'Completed Visits', 'Completion Rate', 'Leave Days', 'Avg Duration (min)'];
            
            default:
                return [];
        }
    }

    public function title(): string
    {
        $titles = [
            'users' => 'User Report',
            'visits' => 'Visit Report',
            'leaves' => 'Leave Report',
            'payroll' => 'Payroll Report',
            'clients' => 'Client Report',
            'performance' => 'Performance Report',
        ];
        
        return $titles[$this->reportType] ?? 'Report';
    }

    public function styles(Worksheet $sheet)
    {
        return [
            1 => ['font' => ['bold' => true, 'color' => ['rgb' => 'FFFFFF']], 'fill' => ['fillType' => 'solid', 'startColor' => ['rgb' => '4F46E5']]],
        ];
    }

    public function columnWidths(): array
    {
        switch ($this->reportType) {
            case 'users':
                return ['A' => 25, 'B' => 30, 'C' => 15, 'D' => 20, 'E' => 15, 'F' => 15, 'G' => 15, 'H' => 15];
            
            case 'visits':
                return ['A' => 20, 'B' => 25, 'C' => 25, 'D' => 20, 'E' => 15, 'F' => 15, 'G' => 15, 'H' => 15, 'I' => 15, 'J' => 30];
            
            case 'leaves':
                return ['A' => 20, 'B' => 15, 'C' => 15, 'D' => 15, 'E' => 15, 'F' => 15, 'G' => 30, 'H' => 20, 'I' => 15];
            
            case 'payroll':
                return ['A' => 20, 'B' => 15, 'C' => 15, 'D' => 15, 'E' => 15, 'F' => 15, 'G' => 15, 'H' => 15, 'I' => 15, 'J' => 15, 'K' => 15, 'L' => 30];
            
            case 'clients':
                return ['A' => 25, 'B' => 25, 'C' => 20, 'D' => 15, 'E' => 15, 'F' => 30, 'G' => 15, 'H' => 25, 'I' => 15, 'J' => 15, 'K' => 15];
            
            case 'performance':
                return ['A' => 25, 'B' => 20, 'C' => 15, 'D' => 15, 'E' => 15, 'F' => 15, 'G' => 20];
            
            default:
                return ['A' => 20];
        }
    }
}