<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Visit;
use App\Models\Client;
use App\Models\LeaveRequest;
use Carbon\Carbon;

class DashboardController extends Controller
{
    public function index()
    {
        $user = auth()->user();
        
        if ($user->hasRole('admin')) {
            return $this->adminDashboard();
        } elseif ($user->hasRole('supervisor')) {
            return $this->supervisorDashboard();
        } else {
            return $this->employeeDashboard();
        }
    }

    private function adminDashboard()
    {
        $stats = [
            'total_employees' => User::role('employee')->count(),
            'total_supervisors' => User::role('supervisor')->count(),
            'total_clients' => Client::count(),
            'pending_visits' => Visit::where('status', 'scheduled')->count(),
            'completed_visits' => Visit::where('status', 'completed')->count(),
            'pending_leaves' => LeaveRequest::where('status', 'pending')->count(),
        ];

        return view('dashboard', compact('stats'));
    }

    private function supervisorDashboard()
    {
        $user = auth()->user();
        $subordinateIds = $user->subordinates()->pluck('id');
        
        $stats = [
            'team_members' => $user->subordinates()->count(),
            'scheduled_visits' => Visit::whereIn('employee_id', $subordinateIds)
                ->where('status', 'scheduled')
                ->count(),
            'completed_today' => Visit::whereIn('employee_id', $subordinateIds)
                ->where('status', 'completed')
                ->whereDate('completed_at', today())
                ->count(),
            'pending_leaves' => LeaveRequest::whereIn('employee_id', $subordinateIds)
                ->where('status', 'pending')
                ->count(),
        ];

        return view('dashboard-supervisor', compact('stats'));
    }

    private function employeeDashboard()
    {
        $user = auth()->user();
        
        $stats = [
            'total_visits' => $user->visits()->count(),
            'todays_visits' => $user->visits()
                ->whereDate('scheduled_date', today())
                ->count(),
            'completed_visits' => $user->visits()
                ->where('status', 'completed')
                ->count(),
            'pending_leaves' => $user->leaveRequests()
                ->where('status', 'pending')
                ->count(),
        ];

        return view('dashboard-employee', compact('stats'));
    }
}