<?php

namespace App\Http\Controllers;

use App\Models\Payroll;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class PayrollController extends Controller
{
    public function index(Request $request)
    {
        $query = Payroll::with(['employee']);
        
        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }
        
        // Filter by employee
        if ($request->filled('employee_id')) {
            $query->where('employee_id', $request->employee_id);
        }
        
        // Filter by month
        if ($request->filled('month')) {
            $month = Carbon::parse($request->month);
            $query->whereMonth('pay_period_start', $month->month)
                  ->whereYear('pay_period_start', $month->year);
        }
        
        $payrolls = $query->latest()->paginate(15);
        $employees = User::role('employee')->get();
        
        return view('payrolls.index', compact('payrolls', 'employees'));
    }

    public function create()
    {
        $employees = User::role('employee')
            ->where('status', 'active')
            ->orderBy('name')
            ->get();
        
        $defaultPayPeriodStart = Carbon::now()->startOfMonth();
        $defaultPayPeriodEnd = Carbon::now()->endOfMonth();
        
        return view('payrolls.create', compact('employees', 'defaultPayPeriodStart', 'defaultPayPeriodEnd'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'employee_id' => 'required|exists:users,id',
            'pay_period_start' => 'required|date',
            'pay_period_end' => 'required|date|after_or_equal:pay_period_start',
            'basic_salary' => 'required|numeric|min:0',
            'allowances' => 'nullable|numeric|min:0',
            'deductions' => 'nullable|numeric|min:0',
            'overtime' => 'nullable|numeric|min:0',
            'bonus' => 'nullable|numeric|min:0',
            'payment_date' => 'nullable|date',
            'notes' => 'nullable|string|max:1000',
        ]);

        $netSalary = $request->basic_salary 
            + ($request->allowances ?? 0)
            + ($request->overtime ?? 0)
            + ($request->bonus ?? 0)
            - ($request->deductions ?? 0);

        try {
            DB::beginTransaction();
            
            Payroll::create([
                'employee_id' => $request->employee_id,
                'pay_period_start' => $request->pay_period_start,
                'pay_period_end' => $request->pay_period_end,
                'basic_salary' => $request->basic_salary,
                'allowances' => $request->allowances ?? 0,
                'deductions' => $request->deductions ?? 0,
                'overtime' => $request->overtime ?? 0,
                'bonus' => $request->bonus ?? 0,
                'net_salary' => $netSalary,
                'status' => 'draft',
                'payment_date' => $request->payment_date,
                'notes' => $request->notes,
            ]);
            
            DB::commit();
            
            return redirect()->route('payrolls.index')
                ->with('success', 'Payroll created successfully.');
                
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Failed to create payroll: ' . $e->getMessage())
                        ->withInput();
        }
    }

    public function show(Payroll $payroll)
    {
        $payroll->load(['employee.department']);
        return view('payrolls.show', compact('payroll'));
    }

    public function edit(Payroll $payroll)
    {
        if ($payroll->status !== 'draft') {
            return redirect()->route('payrolls.index')
                ->with('error', 'Only draft payrolls can be edited.');
        }
        
        $employees = User::role('employee')
            ->where('status', 'active')
            ->orderBy('name')
            ->get();
        
        return view('payrolls.edit', compact('payroll', 'employees'));
    }

    public function update(Request $request, Payroll $payroll)
    {
        if ($payroll->status !== 'draft') {
            return back()->with('error', 'Only draft payrolls can be edited.');
        }
        
        $request->validate([
            'employee_id' => 'required|exists:users,id',
            'pay_period_start' => 'required|date',
            'pay_period_end' => 'required|date|after_or_equal:pay_period_start',
            'basic_salary' => 'required|numeric|min:0',
            'allowances' => 'nullable|numeric|min:0',
            'deductions' => 'nullable|numeric|min:0',
            'overtime' => 'nullable|numeric|min:0',
            'bonus' => 'nullable|numeric|min:0',
            'payment_date' => 'nullable|date',
            'notes' => 'nullable|string|max:1000',
        ]);

        $netSalary = $request->basic_salary 
            + ($request->allowances ?? 0)
            + ($request->overtime ?? 0)
            + ($request->bonus ?? 0)
            - ($request->deductions ?? 0);

        $payroll->update([
            'employee_id' => $request->employee_id,
            'pay_period_start' => $request->pay_period_start,
            'pay_period_end' => $request->pay_period_end,
            'basic_salary' => $request->basic_salary,
            'allowances' => $request->allowances ?? 0,
            'deductions' => $request->deductions ?? 0,
            'overtime' => $request->overtime ?? 0,
            'bonus' => $request->bonus ?? 0,
            'net_salary' => $netSalary,
            'payment_date' => $request->payment_date,
            'notes' => $request->notes,
        ]);

        return redirect()->route('payrolls.index')
            ->with('success', 'Payroll updated successfully.');
    }

    public function destroy(Payroll $payroll)
    {
        if ($payroll->status !== 'draft') {
            return back()->with('error', 'Only draft payrolls can be deleted.');
        }
        
        $payroll->delete();
        
        return redirect()->route('payrolls.index')
            ->with('success', 'Payroll deleted successfully.');
    }

    public function process(Payroll $payroll)
    {
        if ($payroll->status !== 'draft') {
            return back()->with('error', 'Only draft payrolls can be processed.');
        }
        
        $payroll->update([
            'status' => 'processed',
            'processed_at' => now(),
        ]);
        
        return back()->with('success', 'Payroll processed successfully.');
    }

    public function markAsPaid(Payroll $payroll)
    {
        if ($payroll->status !== 'processed') {
            return back()->with('error', 'Only processed payrolls can be marked as paid.');
        }
        
        $payroll->update([
            'status' => 'paid',
            'payment_date' => $payroll->payment_date ?? now(),
        ]);
        
        return back()->with('success', 'Payroll marked as paid.');
    }

    public function generate(Request $request)
    {
        $request->validate([
            'month' => 'required|date_format:Y-m',
            'employee_ids' => 'nullable|array',
            'employee_ids.*' => 'exists:users,id',
        ]);
        
        $month = Carbon::parse($request->month);
        $startDate = $month->copy()->startOfMonth();
        $endDate = $month->copy()->endOfMonth();
        
        $employees = $request->employee_ids 
            ? User::role('employee')->where('status', 'active')->whereIn('id', $request->employee_ids)->get()
            : User::role('employee')->where('status', 'active')->get();
        
        $generatedCount = 0;
        $errors = [];
        
        try {
            DB::beginTransaction();
            
            foreach ($employees as $employee) {
                $existingPayroll = Payroll::where('employee_id', $employee->id)
                    ->where('pay_period_start', $startDate)
                    ->where('pay_period_end', $endDate)
                    ->exists();
                
                if ($existingPayroll) {
                    $errors[] = "Payroll already exists for {$employee->name} ({$month->format('F Y')})";
                    continue;
                }
                
                $basicSalary = $employee->basic_salary ?? 5000;
                
                Payroll::create([
                    'employee_id' => $employee->id,
                    'pay_period_start' => $startDate,
                    'pay_period_end' => $endDate,
                    'basic_salary' => $basicSalary,
                    'allowances' => 0,
                    'deductions' => 0,
                    'overtime' => 0,
                    'bonus' => 0,
                    'net_salary' => $basicSalary,
                    'status' => 'draft',
                    'notes' => 'Auto-generated for ' . $month->format('F Y'),
                ]);
                
                $generatedCount++;
            }
            
            DB::commit();
            
            $message = "Generated {$generatedCount} payroll(s) for {$month->format('F Y')}.";
            if (!empty($errors)) {
                $message .= ' ' . count($errors) . ' error(s) occurred.';
            }
            
            return redirect()->route('payrolls.index')
                ->with('success', $message)
                ->with('errors', $errors);
                
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Failed to generate payrolls: ' . $e->getMessage());
        }
    }

    public function getEmployeeSalary(Request $request)
    {
        $employee = User::find($request->employee_id);
        
        if (!$employee) {
            return response()->json(['error' => 'Employee not found'], 404);
        }
        
        $basicSalary = $employee->basic_salary ?? 5000;
        
        return response()->json([
            'basic_salary' => $basicSalary,
            'currency' => 'EGP',
            'employee_name' => $employee->name,
            'department' => $employee->department->name ?? 'N/A',
        ]);
    }
}