<?php

namespace App\Http\Controllers;

use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;
use Intervention\Image\Facades\Image;

class SettingController extends Controller
{
    public function index()
    {
        $settings = Setting::orderBy('group')
            ->orderBy('sort_order')
            ->get()
            ->groupBy('group');
        
        $groups = [
            'general' => 'General Settings',
            'appearance' => 'Appearance',
            'email' => 'Email Settings',
            'map' => 'Map & GPS Settings',
            'security' => 'Security Settings',
            'payment' => 'Payment Settings',
            'notification' => 'Notifications',
        ];
        
        return view('settings.index', compact('settings', 'groups'));
    }

    public function update(Request $request)
    {
        $settings = $request->except(['_token', '_method']);
        
        foreach ($settings as $key => $value) {
            $setting = Setting::where('key', $key)->first();
            
            if ($setting) {
                // Handle file uploads
                if ($setting->type === 'file' && $request->hasFile($key)) {
                    $file = $request->file($key);
                    
                    // Delete old file if exists
                    if ($setting->value) {
                        Storage::disk('public')->delete($setting->value);
                    }
                    
                    // Upload new file
                    $filename = $key . '-' . time() . '.' . $file->getClientOriginalExtension();
                    $path = $file->storeAs('settings', $filename, 'public');
                    
                    // If it's an image, create thumbnail
                    if (in_array($file->getClientOriginalExtension(), ['jpg', 'jpeg', 'png', 'gif'])) {
                        $this->createImageThumbnail($path, $key);
                    }
                    
                    $value = $path;
                }
                
                $setting->update(['value' => $value]);
            }
        }
        
        // Clear settings cache if you're using caching
        cache()->forget('app_settings');
        
        return redirect()->route('settings.index')
            ->with('success', 'Settings updated successfully.');
    }

    public function general()
    {
        $settings = Setting::where('group', 'general')->get();
        return view('settings.general', compact('settings'));
    }

    public function appearance()
    {
        $settings = Setting::where('group', 'appearance')->get();
        $themes = [
            'light' => 'Light Theme',
            'dark' => 'Dark Theme',
            'blue' => 'Blue Theme',
            'green' => 'Green Theme',
        ];
        
        $layouts = [
            'vertical' => 'Vertical Layout',
            'horizontal' => 'Horizontal Layout',
            'compact' => 'Compact Layout',
        ];
        
        return view('settings.appearance', compact('settings', 'themes', 'layouts'));
    }

    public function email()
    {
        $settings = Setting::where('group', 'email')->get();
        $mailDrivers = [
            'smtp' => 'SMTP',
            'mailgun' => 'Mailgun',
            'ses' => 'Amazon SES',
            'postmark' => 'Postmark',
        ];
        
        return view('settings.email', compact('settings', 'mailDrivers'));
    }

    public function map()
    {
        $settings = Setting::where('group', 'map')->get();
        $mapProviders = [
            'openstreetmap' => 'OpenStreetMap',
            'google_maps' => 'Google Maps',
            'mapbox' => 'Mapbox',
        ];
        
        return view('settings.map', compact('settings', 'mapProviders'));
    }

    public function seedDefaultSettings()
    {
        $defaultSettings = [
            // General Settings
            [
                'key' => 'app_name',
                'value' => 'Innovaxcess Attendance',
                'type' => 'text',
                'group' => 'general',
                'label' => 'Application Name',
                'description' => 'The name of your application',
                'sort_order' => 1,
            ],
            [
                'key' => 'app_description',
                'value' => 'Employee Attendance & Visit Management System',
                'type' => 'textarea',
                'group' => 'general',
                'label' => 'Application Description',
                'description' => 'A brief description of your application',
                'sort_order' => 2,
            ],
            [
                'key' => 'company_name',
                'value' => 'Innovaxcess Inc.',
                'type' => 'text',
                'group' => 'general',
                'label' => 'Company Name',
                'description' => 'Your company/organization name',
                'sort_order' => 3,
            ],
            [
                'key' => 'company_address',
                'value' => '123 Business Street, Cairo, Egypt',
                'type' => 'textarea',
                'group' => 'general',
                'label' => 'Company Address',
                'description' => 'Your company physical address',
                'sort_order' => 4,
            ],
            [
                'key' => 'company_phone',
                'value' => '+20 123 456 7890',
                'type' => 'text',
                'group' => 'general',
                'label' => 'Company Phone',
                'description' => 'Primary contact phone number',
                'sort_order' => 5,
            ],
            [
                'key' => 'company_email',
                'value' => 'info@innovaxcess.com',
                'type' => 'email',
                'group' => 'general',
                'label' => 'Company Email',
                'description' => 'Primary contact email address',
                'sort_order' => 6,
            ],
            [
                'key' => 'timezone',
                'value' => 'Africa/Cairo',
                'type' => 'select',
                'group' => 'general',
                'label' => 'Timezone',
                'description' => 'Default timezone for the application',
                'options' => json_encode([
                    'Africa/Cairo' => 'Cairo, Egypt',
                    'UTC' => 'UTC',
                    'America/New_York' => 'New York, USA',
                    'Europe/London' => 'London, UK',
                    'Asia/Dubai' => 'Dubai, UAE',
                ]),
                'sort_order' => 7,
            ],
            [
                'key' => 'date_format',
                'value' => 'Y-m-d',
                'type' => 'select',
                'group' => 'general',
                'label' => 'Date Format',
                'description' => 'Default date format',
                'options' => json_encode([
                    'Y-m-d' => '2024-01-15',
                    'd/m/Y' => '15/01/2024',
                    'm/d/Y' => '01/15/2024',
                    'F j, Y' => 'January 15, 2024',
                ]),
                'sort_order' => 8,
            ],
            [
                'key' => 'time_format',
                'value' => 'H:i',
                'type' => 'select',
                'group' => 'general',
                'label' => 'Time Format',
                'description' => 'Default time format',
                'options' => json_encode([
                    'H:i' => '14:30 (24-hour)',
                    'h:i A' => '02:30 PM (12-hour)',
                ]),
                'sort_order' => 9,
            ],
            [
                'key' => 'currency',
                'value' => 'EGP',
                'type' => 'select',
                'group' => 'general',
                'label' => 'Currency',
                'description' => 'Default currency for payments',
                'options' => json_encode([
                    'EGP' => 'Egyptian Pound (EGP)',
                    'USD' => 'US Dollar ($)',
                    'EUR' => 'Euro (€)',
                    'GBP' => 'British Pound (£)',
                    'AED' => 'UAE Dirham (AED)',
                ]),
                'sort_order' => 10,
            ],
            [
                'key' => 'currency_symbol',
                'value' => 'EGP',
                'type' => 'text',
                'group' => 'general',
                'label' => 'Currency Symbol',
                'description' => 'Currency symbol to display',
                'sort_order' => 11,
            ],

            // Appearance Settings
            [
                'key' => 'logo',
                'value' => null,
                'type' => 'file',
                'group' => 'appearance',
                'label' => 'Logo',
                'description' => 'Upload your company logo (Recommended: 200x60px)',
                'sort_order' => 1,
            ],
            [
                'key' => 'favicon',
                'value' => null,
                'type' => 'file',
                'group' => 'appearance',
                'label' => 'Favicon',
                'description' => 'Upload favicon (Recommended: 32x32px)',
                'sort_order' => 2,
            ],
            [
                'key' => 'login_background',
                'value' => null,
                'type' => 'file',
                'group' => 'appearance',
                'label' => 'Login Background',
                'description' => 'Background image for login page',
                'sort_order' => 3,
            ],
            [
                'key' => 'theme',
                'value' => 'light',
                'type' => 'select',
                'group' => 'appearance',
                'label' => 'Theme',
                'description' => 'Default color theme',
                'options' => json_encode([
                    'light' => 'Light Theme',
                    'dark' => 'Dark Theme',
                    'blue' => 'Blue Theme',
                    'green' => 'Green Theme',
                ]),
                'sort_order' => 4,
            ],
            [
                'key' => 'layout',
                'value' => 'vertical',
                'type' => 'select',
                'group' => 'appearance',
                'label' => 'Layout',
                'description' => 'Default layout style',
                'options' => json_encode([
                    'vertical' => 'Vertical Layout',
                    'horizontal' => 'Horizontal Layout',
                    'compact' => 'Compact Layout',
                ]),
                'sort_order' => 5,
            ],
            [
                'key' => 'primary_color',
                'value' => '#3b82f6',
                'type' => 'color',
                'group' => 'appearance',
                'label' => 'Primary Color',
                'description' => 'Primary brand color',
                'sort_order' => 6,
            ],
            [
                'key' => 'secondary_color',
                'value' => '#6b7280',
                'type' => 'color',
                'group' => 'appearance',
                'label' => 'Secondary Color',
                'description' => 'Secondary brand color',
                'sort_order' => 7,
            ],

            // Email Settings
            [
                'key' => 'mail_driver',
                'value' => 'smtp',
                'type' => 'select',
                'group' => 'email',
                'label' => 'Mail Driver',
                'description' => 'Email sending method',
                'options' => json_encode([
                    'smtp' => 'SMTP',
                    'mailgun' => 'Mailgun',
                    'ses' => 'Amazon SES',
                    'postmark' => 'Postmark',
                ]),
                'sort_order' => 1,
            ],
            [
                'key' => 'mail_host',
                'value' => 'smtp.mailtrap.io',
                'type' => 'text',
                'group' => 'email',
                'label' => 'SMTP Host',
                'description' => 'SMTP server address',
                'sort_order' => 2,
            ],
            [
                'key' => 'mail_port',
                'value' => '2525',
                'type' => 'number',
                'group' => 'email',
                'label' => 'SMTP Port',
                'description' => 'SMTP server port',
                'sort_order' => 3,
            ],
            [
                'key' => 'mail_username',
                'value' => '',
                'type' => 'text',
                'group' => 'email',
                'label' => 'SMTP Username',
                'description' => 'SMTP authentication username',
                'sort_order' => 4,
            ],
            [
                'key' => 'mail_password',
                'value' => '',
                'type' => 'password',
                'group' => 'email',
                'label' => 'SMTP Password',
                'description' => 'SMTP authentication password',
                'sort_order' => 5,
            ],
            [
                'key' => 'mail_encryption',
                'value' => 'tls',
                'type' => 'select',
                'group' => 'email',
                'label' => 'Encryption',
                'description' => 'SMTP encryption method',
                'options' => json_encode([
                    '' => 'None',
                    'ssl' => 'SSL',
                    'tls' => 'TLS',
                ]),
                'sort_order' => 6,
            ],
            [
                'key' => 'mail_from_address',
                'value' => 'noreply@innovaxcess.com',
                'type' => 'email',
                'group' => 'email',
                'label' => 'From Email Address',
                'description' => 'Default sender email address',
                'sort_order' => 7,
            ],
            [
                'key' => 'mail_from_name',
                'value' => 'Innovaxcess Attendance',
                'type' => 'text',
                'group' => 'email',
                'label' => 'From Name',
                'description' => 'Default sender name',
                'sort_order' => 8,
            ],

            // Map & GPS Settings
            [
                'key' => 'map_provider',
                'value' => 'openstreetmap',
                'type' => 'select',
                'group' => 'map',
                'label' => 'Map Provider',
                'description' => 'Default map service provider',
                'options' => json_encode([
                    'openstreetmap' => 'OpenStreetMap (Free)',
                    'google_maps' => 'Google Maps',
                    'mapbox' => 'Mapbox',
                ]),
                'sort_order' => 1,
            ],
            [
                'key' => 'google_maps_api_key',
                'value' => '',
                'type' => 'text',
                'group' => 'map',
                'label' => 'Google Maps API Key',
                'description' => 'Required if using Google Maps',
                'sort_order' => 2,
            ],
            [
                'key' => 'mapbox_access_token',
                'value' => '',
                'type' => 'text',
                'group' => 'map',
                'label' => 'Mapbox Access Token',
                'description' => 'Required if using Mapbox',
                'sort_order' => 3,
            ],
            [
                'key' => 'default_latitude',
                'value' => '30.0444',
                'type' => 'number',
                'group' => 'map',
                'label' => 'Default Latitude',
                'description' => 'Default map center latitude',
                'sort_order' => 4,
            ],
            [
                'key' => 'default_longitude',
                'value' => '31.2357',
                'type' => 'number',
                'group' => 'map',
                'label' => 'Default Longitude',
                'description' => 'Default map center longitude',
                'sort_order' => 5,
            ],
            [
                'key' => 'default_zoom',
                'value' => '12',
                'type' => 'number',
                'group' => 'map',
                'label' => 'Default Zoom Level',
                'description' => 'Default map zoom level (1-20)',
                'sort_order' => 6,
            ],
            [
                'key' => 'gps_update_interval',
                'value' => '30',
                'type' => 'number',
                'group' => 'map',
                'label' => 'GPS Update Interval',
                'description' => 'GPS location update interval in seconds',
                'sort_order' => 7,
            ],
            [
                'key' => 'visit_proximity_radius',
                'value' => '100',
                'type' => 'number',
                'group' => 'map',
                'label' => 'Visit Proximity Radius',
                'description' => 'Radius in meters for auto-completing visits',
                'sort_order' => 8,
            ],

            // Security Settings
            [
                'key' => 'login_attempts',
                'value' => '5',
                'type' => 'number',
                'group' => 'security',
                'label' => 'Max Login Attempts',
                'description' => 'Maximum failed login attempts before lockout',
                'sort_order' => 1,
            ],
            [
                'key' => 'lockout_time',
                'value' => '15',
                'type' => 'number',
                'group' => 'security',
                'label' => 'Lockout Time (minutes)',
                'description' => 'Account lockout duration in minutes',
                'sort_order' => 2,
            ],
            [
                'key' => 'session_timeout',
                'value' => '30',
                'type' => 'number',
                'group' => 'security',
                'label' => 'Session Timeout (minutes)',
                'description' => 'User session timeout in minutes',
                'sort_order' => 3,
            ],
            [
                'key' => 'password_min_length',
                'value' => '8',
                'type' => 'number',
                'group' => 'security',
                'label' => 'Minimum Password Length',
                'description' => 'Minimum characters required for passwords',
                'sort_order' => 4,
            ],
            [
                'key' => 'require_mixed_case',
                'value' => '1',
                'type' => 'checkbox',
                'group' => 'security',
                'label' => 'Require Mixed Case',
                'description' => 'Require both uppercase and lowercase letters',
                'sort_order' => 5,
            ],
            [
                'key' => 'require_numbers',
                'value' => '1',
                'type' => 'checkbox',
                'group' => 'security',
                'label' => 'Require Numbers',
                'description' => 'Require at least one number',
                'sort_order' => 6,
            ],
            [
                'key' => 'require_special_chars',
                'value' => '0',
                'type' => 'checkbox',
                'group' => 'security',
                'label' => 'Require Special Characters',
                'description' => 'Require at least one special character',
                'sort_order' => 7,
            ],
            [
                'key' => 'password_expiry_days',
                'value' => '90',
                'type' => 'number',
                'group' => 'security',
                'label' => 'Password Expiry (days)',
                'description' => 'Days before password expires (0 for never)',
                'sort_order' => 8,
            ],

            // Payment Settings
            [
                'key' => 'payment_currency',
                'value' => 'EGP',
                'type' => 'select',
                'group' => 'payment',
                'label' => 'Payment Currency',
                'description' => 'Currency for payroll payments',
                'options' => json_encode([
                    'EGP' => 'Egyptian Pound',
                    'USD' => 'US Dollar',
                    'EUR' => 'Euro',
                ]),
                'sort_order' => 1,
            ],
            [
                'key' => 'default_basic_salary',
                'value' => '5000',
                'type' => 'number',
                'group' => 'payment',
                'label' => 'Default Basic Salary',
                'description' => 'Default basic salary for new employees',
                'sort_order' => 2,
            ],
            [
                'key' => 'overtime_rate',
                'value' => '1.5',
                'type' => 'number',
                'group' => 'payment',
                'label' => 'Overtime Rate',
                'description' => 'Overtime multiplier (e.g., 1.5 = time and a half)',
                'sort_order' => 3,
            ],
            [
                'key' => 'tax_percentage',
                'value' => '10',
                'type' => 'number',
                'group' => 'payment',
                'label' => 'Tax Percentage',
                'description' => 'Default tax percentage for payroll',
                'sort_order' => 4,
            ],
            [
                'key' => 'social_insurance_percentage',
                'value' => '14',
                'type' => 'number',
                'group' => 'payment',
                'label' => 'Social Insurance Percentage',
                'description' => 'Social insurance deduction percentage',
                'sort_order' => 5,
            ],

            // Notification Settings
            [
                'key' => 'notify_new_visit',
                'value' => '1',
                'type' => 'checkbox',
                'group' => 'notification',
                'label' => 'Notify New Visit',
                'description' => 'Send notification when new visit is assigned',
                'sort_order' => 1,
            ],
            [
                'key' => 'notify_visit_start',
                'value' => '1',
                'type' => 'checkbox',
                'group' => 'notification',
                'label' => 'Notify Visit Start',
                'description' => 'Send notification when visit starts',
                'sort_order' => 2,
            ],
            [
                'key' => 'notify_visit_complete',
                'value' => '1',
                'type' => 'checkbox',
                'group' => 'notification',
                'label' => 'Notify Visit Complete',
                'description' => 'Send notification when visit completes',
                'sort_order' => 3,
            ],
            [
                'key' => 'notify_new_leave',
                'value' => '1',
                'type' => 'checkbox',
                'group' => 'notification',
                'label' => 'Notify New Leave Request',
                'description' => 'Send notification for new leave requests',
                'sort_order' => 4,
            ],
            [
                'key' => 'notify_leave_approval',
                'value' => '1',
                'type' => 'checkbox',
                'group' => 'notification',
                'label' => 'Notify Leave Approval/Rejection',
                'description' => 'Send notification when leave is approved/rejected',
                'sort_order' => 5,
            ],
            [
                'key' => 'notify_payroll_processed',
                'value' => '1',
                'type' => 'checkbox',
                'group' => 'notification',
                'label' => 'Notify Payroll Processed',
                'description' => 'Send notification when payroll is processed',
                'sort_order' => 6,
            ],
            [
                'key' => 'notify_payroll_paid',
                'value' => '1',
                'type' => 'checkbox',
                'group' => 'notification',
                'label' => 'Notify Payroll Paid',
                'description' => 'Send notification when payroll is paid',
                'sort_order' => 7,
            ],
        ];

        foreach ($defaultSettings as $setting) {
            Setting::firstOrCreate(
                ['key' => $setting['key']],
                $setting
            );
        }

        return redirect()->route('settings.index')
            ->with('success', 'Default settings seeded successfully.');
    }

    private function createImageThumbnail($path, $key)
    {
        try {
            $fullPath = storage_path('app/public/' . $path);
            
            if (file_exists($fullPath)) {
                $image = Image::make($fullPath);
                
                // Different thumbnail sizes based on setting type
                if ($key === 'logo') {
                    $image->resize(200, 60, function ($constraint) {
                        $constraint->aspectRatio();
                        $constraint->upsize();
                    });
                } elseif ($key === 'favicon') {
                    $image->resize(32, 32);
                } elseif ($key === 'login_background') {
                    $image->resize(1920, 1080, function ($constraint) {
                        $constraint->aspectRatio();
                        $constraint->upsize();
                    });
                }
                
                $image->save($fullPath);
            }
        } catch (\Exception $e) {
            // Log error but don't break the process
            \Log::error('Failed to create thumbnail: ' . $e->getMessage());
        }
    }

    public function testEmail()
    {
        try {
            \Mail::raw('Test email from Innovaxcess Attendance System', function ($message) {
                $message->to(auth()->user()->email)
                        ->subject('Email Test - Innovaxcess Attendance');
            });
            
            return back()->with('success', 'Test email sent successfully to ' . auth()->user()->email);
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to send test email: ' . $e->getMessage());
        }
    }

    public function clearCache()
    {
        \Artisan::call('cache:clear');
        \Artisan::call('config:clear');
        \Artisan::call('view:clear');
        
        return back()->with('success', 'Application cache cleared successfully.');
    }
}