<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Role;
use App\Models\Department;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;

class UserController extends Controller
{
    public function index()
    {
        $users = User::with(['department', 'supervisor', 'roles'])->paginate(10);
        
        // Calculate counts for stats
        $activeUsersCount = User::where('status', 'active')->count();
        $adminUsersCount = User::role('admin')->count();
        $supervisorUsersCount = User::role('supervisor')->count();
        
        return view('users.index', compact('users', 'activeUsersCount', 'adminUsersCount', 'supervisorUsersCount'));
    }

    public function create()
    {
        $roles = Role::all();
        $departments = Department::where('status', 'active')->get();
        $supervisors = User::role('supervisor')->get();
        return view('users.create', compact('roles', 'departments', 'supervisors'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users',
            'password' => 'required|min:8',
            'phone' => 'required|string',
            'employee_id' => 'required|unique:users',
            'department_id' => 'required|exists:departments,id',
            'role' => 'required|exists:roles,name',
        ]);

        DB::beginTransaction();
        try {
            $user = User::create([
                'name' => $request->name,
                'email' => $request->email,
                'password' => Hash::make($request->password),
                'phone' => $request->phone,
                'employee_id' => $request->employee_id,
                'department_id' => $request->department_id,
                'supervisor_id' => $request->supervisor_id,
                'address' => $request->address,
                'hire_date' => $request->hire_date,
                'status' => 'active',
            ]);

            $user->assignRole($request->role);
            
            DB::commit();
            return redirect()->route('users.index')->with('success', 'User created successfully.');
            
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Error creating user: ' . $e->getMessage())->withInput();
        }
    }

    public function show($id)
    {
        $user = User::with(['department', 'supervisor', 'roles'])->findOrFail($id);
        return view('users.show', compact('user'));
    }

    public function edit($id)
    {
        $user = User::findOrFail($id);
        $roles = Role::all();
        $departments = Department::where('status', 'active')->get();
        $supervisors = User::role('supervisor')->where('id', '!=', $id)->get();
        return view('users.edit', compact('user', 'roles', 'departments', 'supervisors'));
    }

    public function update(Request $request, $id)
    {
        $user = User::findOrFail($id);
        
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $user->id,
            'phone' => 'required|string',
            'employee_id' => 'required|unique:users,employee_id,' . $user->id,
            'department_id' => 'required|exists:departments,id',
            'role' => 'required|exists:roles,name',
        ]);

        DB::beginTransaction();
        try {
            $user->update([
                'name' => $request->name,
                'email' => $request->email,
                'phone' => $request->phone,
                'employee_id' => $request->employee_id,
                'department_id' => $request->department_id,
                'supervisor_id' => $request->supervisor_id,
                'address' => $request->address,
                'hire_date' => $request->hire_date,
                'status' => $request->status ?? $user->status,
            ]);

            if ($request->password) {
                $user->update(['password' => Hash::make($request->password)]);
            }

            $user->syncRoles([$request->role]);
            
            DB::commit();
            return redirect()->route('users.index')->with('success', 'User updated successfully.');
            
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Error updating user: ' . $e->getMessage())->withInput();
        }
    }

    public function destroy($id)
    {
        $user = User::findOrFail($id);
        
        // Prevent deleting yourself
        if ($user->id === auth()->id()) {
            return redirect()->route('users.index')->with('error', 'You cannot delete your own account.');
        }
        
        $user->delete();
        return redirect()->route('users.index')->with('success', 'User deleted successfully.');
    }

    public function deactivate($id)
    {
        $user = User::findOrFail($id);
        
        // Prevent deactivating yourself
        if ($user->id === auth()->id()) {
            return redirect()->route('users.index')->with('error', 'You cannot deactivate your own account.');
        }
        
        $user->update(['status' => 'inactive']);
        return redirect()->route('users.index')->with('success', 'User deactivated successfully.');
    }

    public function activate($id)
    {
        $user = User::findOrFail($id);
        $user->update(['status' => 'active']);
        return redirect()->route('users.index')->with('success', 'User activated successfully.');
    }
}