<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Symfony\Component\HttpFoundation\Response;

class RedirectBasedOnRole
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next): Response
    {
        if (Auth::check()) {
            $user = Auth::user();
            
            // If trying to access login page and already logged in, redirect based on role
            if ($request->routeIs('login')) {
                return $this->redirectToDashboard($user);
            }
            
            // Check if user has access to the requested route
            if (!$this->hasAccess($user, $request)) {
                return $this->redirectToDashboard($user);
            }
        }
        
        return $next($request);
    }
    
    private function redirectToDashboard($user)
    {
        if ($user->isAdmin()) {
            return redirect()->route('dashboard');
        } elseif ($user->isSupervisor()) {
            return redirect()->route('dashboard');
        } elseif ($user->isEmployee()) {
            return redirect()->route('employee.dashboard');
        }
        
        return redirect('/');
    }
    
    private function hasAccess($user, $request)
    {
        $route = $request->route()->getName();
        
        // Define accessible routes for each role
        $adminRoutes = [
            'dashboard',
            'users.*',
            'departments.*',
            'clients.*',
            'visits.*',
            'live-tracking.*',
            'leaves.*',
            'payrolls.*',
            'reports.*',
            'settings.*',
            'profile.*',
            'logout'
        ];
        
        $supervisorRoutes = [
            'dashboard',
            'visits.*',
            'live-tracking.*',
            'leaves.index',
            'leaves.create',
            'leaves.show',
            'profile.*',
            'logout'
        ];
        
        $employeeRoutes = [
            'employee.dashboard',
            'visits.index',
            'visits.show',
            'live-tracking.*',
            'leaves.index',
            'leaves.create',
            'leaves.show',
            'profile.*',
            'logout'
        ];
        
        // Check if route matches allowed patterns
        $checkRoute = function($patterns, $route) {
            foreach ($patterns as $pattern) {
                if (str_ends_with($pattern, '.*')) {
                    $prefix = str_replace('.*', '', $pattern);
                    if (str_starts_with($route, $prefix)) {
                        return true;
                    }
                } elseif ($pattern === $route) {
                    return true;
                }
            }
            return false;
        };
        
        if ($user->isAdmin()) {
            return $checkRoute($adminRoutes, $route);
        } elseif ($user->isSupervisor()) {
            return $checkRoute($supervisorRoutes, $route);
        } elseif ($user->isEmployee()) {
            return $checkRoute($employeeRoutes, $route);
        }
        
        return false;
    }
}