<?php

namespace App\Models;

use Spatie\Permission\Models\Role as SpatieRole;

class Role extends SpatieRole
{
    protected $fillable = [
        'name', 
        'guard_name', 
        'description'
    ];

    // ONLY add methods that DON'T exist in the parent SpatieRole class
    // DO NOT override any methods that already exist in SpatieRole
    
    // Custom scope for active users
    public function activeUsers()
    {
        return $this->users()->where('status', 'active');
    }

    // Custom scope for inactive users
    public function inactiveUsers()
    {
        return $this->users()->where('status', '!=', 'active');
    }

    // Scope for web guard
    public function scopeWeb($query)
    {
        return $query->where('guard_name', 'web');
    }

    // Scope for api guard
    public function scopeApi($query)
    {
        return $query->where('guard_name', 'api');
    }

    // Custom attribute for display name
    public function getDisplayNameAttribute()
    {
        return ucfirst(str_replace('_', ' ', $this->name));
    }

    // Custom attribute for statistics
    public function getStatisticsAttribute()
    {
        return [
            'total_users' => $this->users()->count(),
            'active_users' => $this->activeUsers()->count(),
            'inactive_users' => $this->inactiveUsers()->count(),
            'permission_count' => $this->permissions()->count(),
        ];
    }

    // Get permissions grouped by category (custom method with unique name)
    public function getGroupedPermissions()
    {
        $permissions = $this->permissions;
        $grouped = [];
        
        foreach ($permissions as $permission) {
            $parts = explode('_', $permission->name);
            $group = $parts[0] ?? 'other';
            
            if (!isset($grouped[$group])) {
                $grouped[$group] = [];
            }
            
            $grouped[$group][] = $permission;
        }
        
        return $grouped;
    }

    // Check role hierarchy (custom method with unique name)
    public function canManageOtherRole($otherRole)
    {
        if (!$otherRole instanceof Role) {
            $otherRole = Role::where('name', $otherRole)->first();
        }
        
        if (!$otherRole) return false;
        
        // Define hierarchy levels
        $hierarchy = [
            'super_admin' => 1,
            'admin' => 2,
            'supervisor' => 3,
            'employee' => 4
        ];
        
        $thisLevel = $hierarchy[$this->name] ?? 5;
        $otherLevel = $hierarchy[$otherRole->name] ?? 5;
        
        return $thisLevel < $otherLevel;
    }

    // Custom attribute for hierarchy level
    public function getHierarchyLevelAttribute()
    {
        $hierarchy = [
            'super_admin' => 1,
            'admin' => 2,
            'supervisor' => 3,
            'employee' => 4
        ];
        
        return $hierarchy[$this->name] ?? 5;
    }

    // Check if this is a system role
    public function getIsSystemRoleAttribute()
    {
        $systemRoles = ['admin', 'super_admin', 'supervisor', 'employee'];
        return in_array($this->name, $systemRoles);
    }
}