@extends('layouts.dashboard')

@section('title', 'Tracking History - ' . $employee->name)

@section('content')
<div class="container mx-auto px-4">
    <div class="flex justify-between items-center mb-6">
        <div>
            <h1 class="text-2xl font-bold text-gray-800">Tracking History</h1>
            <p class="text-gray-600">for {{ $employee->name }} ({{ $employee->employee_id }})</p>
        </div>
        <div class="flex space-x-3">
            <a href="{{ route('live-tracking.employees') }}" class="bg-gray-600 hover:bg-gray-700 text-white font-bold py-2 px-4 rounded flex items-center">
                <i class="fas fa-arrow-left mr-2"></i> Back to Employees
            </a>
            <a href="{{ route('live-tracking.map') }}" class="bg-blue-600 hover:bg-blue-700 text-white font-bold py-2 px-4 rounded flex items-center">
                <i class="fas fa-map mr-2"></i> View Map
            </a>
        </div>
    </div>

    <!-- Employee Info Card -->
    <div class="bg-white shadow rounded-lg p-6 mb-6">
        <div class="flex items-center">
            <div class="flex-shrink-0 h-16 w-16">
                <div class="h-16 w-16 rounded-full bg-blue-600 flex items-center justify-center text-white text-xl">
                    @if($employee->profile_picture)
                        <img src="{{ Storage::url($employee->profile_picture) }}" 
                             alt="{{ $employee->name }}" 
                             class="h-16 w-16 rounded-full">
                    @else
                        <i class="fas fa-user"></i>
                    @endif
                </div>
            </div>
            <div class="ml-6">
                <h2 class="text-xl font-bold text-gray-900">{{ $employee->name }}</h2>
                <div class="grid grid-cols-2 md:grid-cols-4 gap-4 mt-2">
                    <div>
                        <p class="text-sm text-gray-600">Employee ID</p>
                        <p class="font-medium">{{ $employee->employee_id }}</p>
                    </div>
                    <div>
                        <p class="text-sm text-gray-600">Department</p>
                        <p class="font-medium">{{ $employee->department->name ?? 'N/A' }}</p>
                    </div>
                    <div>
                        <p class="text-sm text-gray-600">Phone</p>
                        <p class="font-medium">{{ $employee->phone ?? 'N/A' }}</p>
                    </div>
                    <div>
                        <p class="text-sm text-gray-600">Total Visits</p>
                        <p class="font-medium">{{ $visits->total() }}</p>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Stats -->
    <div class="grid grid-cols-1 md:grid-cols-4 gap-6 mb-6">
        <div class="bg-white shadow rounded-lg p-6">
            <div class="flex items-center">
                <div class="p-3 rounded-full bg-green-100 text-green-600 mr-4">
                    <i class="fas fa-check-circle text-xl"></i>
                </div>
                <div>
                    <p class="text-sm text-gray-600">Completed Visits</p>
                    <h3 class="text-2xl font-bold">{{ $visits->where('status', 'completed')->count() }}</h3>
                </div>
            </div>
        </div>
        
        <div class="bg-white shadow rounded-lg p-6">
            <div class="flex items-center">
                <div class="p-3 rounded-full bg-yellow-100 text-yellow-600 mr-4">
                    <i class="fas fa-spinner text-xl"></i>
                </div>
                <div>
                    <p class="text-sm text-gray-600">In Progress</p>
                    <h3 class="text-2xl font-bold">{{ $visits->where('status', 'in_progress')->count() }}</h3>
                </div>
            </div>
        </div>
        
        <div class="bg-white shadow rounded-lg p-6">
            <div class="flex items-center">
                <div class="p-3 rounded-full bg-blue-100 text-blue-600 mr-4">
                    <i class="fas fa-road text-xl"></i>
                </div>
                <div>
                    <p class="text-sm text-gray-600">Total Distance</p>
                    <h3 class="text-2xl font-bold">{{ round($visits->sum('distance_traveled') / 1000, 1) }} km</h3>
                </div>
            </div>
        </div>
        
        <div class="bg-white shadow rounded-lg p-6">
            <div class="flex items-center">
                <div class="p-3 rounded-full bg-purple-100 text-purple-600 mr-4">
                    <i class="fas fa-clock text-xl"></i>
                </div>
                <div>
                    <p class="text-sm text-gray-600">Avg. Duration</p>
                    <h3 class="text-2xl font-bold">
                        @php
                            $completedVisits = $visits->where('status', 'completed');
                            $totalMinutes = $completedVisits->sum(function($visit) {
                                return $visit->started_at && $visit->completed_at 
                                    ? $visit->started_at->diffInMinutes($visit->completed_at)
                                    : 0;
                            });
                            echo $completedVisits->count() > 0 
                                ? round($totalMinutes / $completedVisits->count()) . ' min'
                                : 'N/A';
                        @endphp
                    </h3>
                </div>
            </div>
        </div>
    </div>

    <!-- Visits Table -->
    <div class="bg-white shadow-md rounded-lg overflow-hidden">
        <div class="px-6 py-4 border-b border-gray-200">
            <h2 class="text-lg font-semibold text-gray-800">Visit History</h2>
        </div>
        <div class="overflow-x-auto">
            <table class="min-w-full divide-y divide-gray-200">
                <thead class="bg-gray-50">
                    <tr>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Date & Time</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Client</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Location</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Duration</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Distance</th>
                        <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                    </tr>
                </thead>
                <tbody class="bg-white divide-y divide-gray-200">
                    @forelse($visits as $visit)
                    <tr>
                        <td class="px-6 py-4 whitespace-nowrap">
                            <div class="text-sm font-medium text-gray-900">{{ $visit->scheduled_date->format('Y-m-d') }}</div>
                            <div class="text-sm text-gray-500">{{ $visit->scheduled_time }}</div>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap">
                            <div class="text-sm font-medium text-gray-900">{{ $visit->client->name }}</div>
                            <div class="text-sm text-gray-500">{{ $visit->client->contact_person ?? 'N/A' }}</div>
                        </td>
                        <td class="px-6 py-4">
                            <div class="text-sm text-gray-900">
                                <i class="fas fa-map-marker-alt text-red-500 mr-1"></i>
                                {{ $visit->client->area ?? 'N/A' }}
                            </div>
                            @if($visit->destination_address)
                                <div class="text-xs text-gray-500 mt-1">
                                    {{ Str::limit($visit->destination_address, 30) }}
                                </div>
                            @endif
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap">
                            @if($visit->status == 'completed')
                                <span class="px-2 py-1 inline-flex text-xs leading-5 font-semibold rounded-full bg-green-100 text-green-800">
                                    <i class="fas fa-check-circle mr-1"></i> Completed
                                </span>
                            @elseif($visit->status == 'in_progress')
                                <span class="px-2 py-1 inline-flex text-xs leading-5 font-semibold rounded-full bg-yellow-100 text-yellow-800">
                                    <i class="fas fa-spinner fa-spin mr-1"></i> In Progress
                                </span>
                            @else
                                <span class="px-2 py-1 inline-flex text-xs leading-5 font-semibold rounded-full bg-gray-100 text-gray-800">
                                    <i class="fas fa-clock mr-1"></i> {{ ucfirst($visit->status) }}
                                </span>
                            @endif
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                            @if($visit->started_at && $visit->completed_at)
                                {{ $visit->started_at->diffForHumans($visit->completed_at, true) }}
                            @elseif($visit->started_at)
                                {{ $visit->started_at->diffForHumans(null, true) }} (ongoing)
                            @else
                                N/A
                            @endif
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                            @if($visit->distance_traveled)
                                {{ round($visit->distance_traveled / 1000, 1) }} km
                            @else
                                N/A
                            @endif
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                            <div class="flex space-x-2">
                                <a href="{{ route('visits.show', $visit) }}" 
                                   class="text-blue-600 hover:text-blue-900"
                                   title="View Details">
                                    <i class="fas fa-eye"></i>
                                </a>
                                @if($visit->status == 'in_progress')
                                <a href="{{ route('gps.track', $visit) }}" 
                                   class="text-green-600 hover:text-green-900"
                                   title="Live Track">
                                    <i class="fas fa-satellite"></i>
                                </a>
                                @endif
                                @if($visit->checkpoints)
                                <button onclick="showRoute({{ $visit->id }})" 
                                        class="text-purple-600 hover:text-purple-900"
                                        title="View Route">
                                    <i class="fas fa-route"></i>
                                </button>
                                @endif
                            </div>
                        </td>
                    </tr>
                    @empty
                    <tr>
                        <td colspan="7" class="px-6 py-8 text-center text-gray-500">
                            <i class="fas fa-history text-3xl mb-3"></i>
                            <p>No tracking history found for this employee</p>
                        </td>
                    </tr>
                    @endforelse
                </tbody>
            </table>
        </div>
        
        <!-- Pagination -->
        @if($visits->hasPages())
        <div class="bg-white px-4 py-3 border-t border-gray-200 sm:px-6">
            {{ $visits->links() }}
        </div>
        @endif
    </div>
</div>

<!-- Route Modal -->
<div id="routeModal" class="modal hidden">
    <div class="modal-content" style="max-width: 800px;">
        <div class="modal-header">
            <h3 class="text-lg font-semibold">Visit Route</h3>
            <button onclick="closeModal()" class="text-gray-400 hover:text-gray-600">
                <i class="fas fa-times"></i>
            </button>
        </div>
        <div class="modal-body">
            <div id="routeMap" style="height: 400px; border-radius: 0.375rem;"></div>
            <div id="routeDetails" class="mt-4"></div>
        </div>
    </div>
</div>

<script>
// Show route on map
function showRoute(visitId) {
    fetch(`/gps/${visitId}/route-history`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                displayRoute(data.route, data.visit);
                openModal();
            } else {
                alert('Route data not available');
            }
        });
}

// Display route on map
function displayRoute(route, visitInfo) {
    const modal = document.getElementById('routeModal');
    const mapDiv = document.getElementById('routeMap');
    const detailsDiv = document.getElementById('routeDetails');
    
    // Clear previous map
    mapDiv.innerHTML = '';
    
    // Create map
    const map = L.map(mapDiv).setView([30.0444, 31.2357], 12);
    
    L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
        attribution: '&copy; OpenStreetMap contributors'
    }).addTo(map);
    
    // Add start marker
    if (visitInfo.start_location.latitude) {
        L.marker([visitInfo.start_location.latitude, visitInfo.start_location.longitude], {
            icon: L.divIcon({
                className: 'employee-marker',
                html: '<i class="fas fa-play" style="color: green;"></i>',
                iconSize: [30, 30],
                iconAnchor: [15, 30]
            })
        }).addTo(map).bindPopup('Start Location');
    }
    
    // Add destination marker
    if (visitInfo.destination.latitude) {
        L.marker([visitInfo.destination.latitude, visitInfo.destination.longitude], {
            icon: L.divIcon({
                className: 'employee-marker',
                html: '<i class="fas fa-flag" style="color: red;"></i>',
                iconSize: [30, 30],
                iconAnchor: [15, 30]
            })
        }).addTo(map).bindPopup('Destination');
    }
    
    // Add route if checkpoints exist
    if (route && route.length > 0) {
        const latlngs = route.map(point => [point.latitude, point.longitude]);
        const polyline = L.polyline(latlngs, {color: 'blue'}).addTo(map);
        map.fitBounds(polyline.getBounds());
        
        // Add checkpoints
        route.forEach((point, index) => {
            L.marker([point.latitude, point.longitude]).addTo(map)
                .bindPopup(`Checkpoint ${index + 1}<br>${point.timestamp}`);
        });
        
        // Calculate total distance
        let totalDistance = 0;
        for (let i = 1; i < latlngs.length; i++) {
            totalDistance += map.distance(latlngs[i-1], latlngs[i]);
        }
        
        detailsDiv.innerHTML = `
            <div class="grid grid-cols-2 gap-4">
                <div>
                    <p class="text-sm text-gray-600">Checkpoints</p>
                    <p class="font-medium">${route.length}</p>
                </div>
                <div>
                    <p class="text-sm text-gray-600">Total Distance</p>
                    <p class="font-medium">${(totalDistance / 1000).toFixed(2)} km</p>
                </div>
                <div>
                    <p class="text-sm text-gray-600">Start Time</p>
                    <p class="font-medium">${route[0]?.timestamp || 'N/A'}</p>
                </div>
                <div>
                    <p class="text-sm text-gray-600">End Time</p>
                    <p class="font-medium">${route[route.length-1]?.timestamp || 'N/A'}</p>
                </div>
            </div>
        `;
    } else {
        detailsDiv.innerHTML = '<p class="text-gray-500">No detailed route data available</p>';
    }
}

// Modal functions
function openModal() {
    document.getElementById('routeModal').classList.remove('hidden');
}

function closeModal() {
    document.getElementById('routeModal').classList.add('hidden');
}

// Close modal on outside click
document.getElementById('routeModal').addEventListener('click', function(e) {
    if (e.target === this) {
        closeModal();
    }
});
</script>

<style>
.modal {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0,0,0,0.5);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 1000;
}

.modal.hidden {
    display: none;
}

.modal-content {
    background: white;
    border-radius: 0.5rem;
    padding: 0;
    max-width: 90%;
    max-height: 90vh;
    overflow-y: auto;
}

.modal-header {
    padding: 1rem 1.5rem;
    border-bottom: 1px solid #e5e7eb;
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.modal-body {
    padding: 1.5rem;
}
</style>
@endsection