@extends('layouts.dashboard')

@section('title', 'Create Payroll')

@section('content')
<div class="container mx-auto px-4 py-6">
    <div class="max-w-6xl mx-auto">
        <!-- Header -->
        <div class="mb-8">
            <div class="flex flex-col md:flex-row md:items-center md:justify-between">
                <div>
                    <a href="{{ route('payrolls.index') }}" 
                       class="text-blue-600 hover:text-blue-700 transition-colors duration-200 inline-flex items-center mb-2 group">
                        <i class="fas fa-arrow-left mr-2 group-hover:-translate-x-1 transition-transform duration-200"></i> 
                        Back to Payrolls
                    </a>
                    <h1 class="text-2xl font-bold text-gray-900">Create New Payroll</h1>
                    <p class="text-gray-600 mt-1">Create a payroll record for an employee</p>
                </div>
                <div class="flex items-center space-x-2 mt-4 md:mt-0">
                    <div class="w-12 h-12 rounded-xl bg-gradient-to-br from-blue-100 to-indigo-200 flex items-center justify-center shadow-sm">
                        <i class="fas fa-money-check-alt text-blue-600 text-xl"></i>
                    </div>
                </div>
            </div>
        </div>

        <!-- Form -->
        <div class="bg-white rounded-xl shadow-sm border border-gray-200 overflow-hidden">
            <!-- Form Header -->
            <div class="px-6 py-5 border-b border-gray-200 bg-gradient-to-r from-blue-50 to-indigo-50">
                <h2 class="text-lg font-semibold text-gray-900">Payroll Information</h2>
                <p class="text-sm text-gray-600 mt-1">Fill in the payroll details for the employee</p>
            </div>

            <form action="{{ route('payrolls.store') }}" method="POST" id="payrollForm" class="p-6">
                @csrf
                
                <!-- Employee Selection -->
                <div class="mb-8">
                    <div class="flex items-center mb-4">
                        <div class="w-8 h-8 rounded-lg bg-blue-100 text-blue-600 flex items-center justify-center mr-3">
                            <i class="fas fa-user-tie"></i>
                        </div>
                        <h3 class="text-lg font-semibold text-gray-900">Employee Information</h3>
                    </div>
                    
                    <div>
                        <label for="employee_id" class="block text-sm font-medium text-gray-700 mb-2">
                            <i class="fas fa-user mr-2 text-blue-500"></i>Select Employee *
                        </label>
                        <select name="employee_id" 
                                id="employee_id" 
                                class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent transition duration-200 @error('employee_id') border-red-500 @enderror"
                                required
                                onchange="fetchEmployeeSalary()">
                            <option value="">Select an employee</option>
                            @foreach($employees as $employee)
                            <option value="{{ $employee->id }}" {{ old('employee_id') == $employee->id ? 'selected' : '' }}>
                                {{ $employee->name }} ({{ $employee->employee_id ?? 'N/A' }})
                                @if($employee->department)
                                 - {{ $employee->department->name }}
                                @endif
                            </option>
                            @endforeach
                        </select>
                        @error('employee_id')
                        <p class="mt-2 text-sm text-red-600 flex items-center">
                            <i class="fas fa-exclamation-circle mr-2"></i> {{ $message }}
                        </p>
                        @enderror
                        
                        <!-- Employee Info Display -->
                        <div id="employeeInfo" class="mt-4 p-5 bg-gradient-to-r from-blue-50 to-indigo-50 rounded-xl border border-blue-100 hidden">
                            <div class="grid grid-cols-1 md:grid-cols-3 gap-5">
                                <div class="p-3 bg-white rounded-lg border border-gray-200">
                                    <p class="text-xs font-medium text-gray-500 mb-1">Department</p>
                                    <p id="employeeDept" class="text-sm font-semibold text-gray-900">-</p>
                                </div>
                                <div class="p-3 bg-white rounded-lg border border-gray-200">
                                    <p class="text-xs font-medium text-gray-500 mb-1">Basic Salary</p>
                                    <p id="employeeSalary" class="text-sm font-semibold text-gray-900">-</p>
                                </div>
                                <div class="p-3 bg-white rounded-lg border border-gray-200">
                                    <p class="text-xs font-medium text-gray-500 mb-1">Currency</p>
                                    <p id="employeeCurrency" class="text-sm font-semibold text-gray-900">-</p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Pay Period -->
                <div class="mb-8">
                    <div class="flex items-center mb-4">
                        <div class="w-8 h-8 rounded-lg bg-indigo-100 text-indigo-600 flex items-center justify-center mr-3">
                            <i class="fas fa-calendar-alt"></i>
                        </div>
                        <h3 class="text-lg font-semibold text-gray-900">Pay Period</h3>
                    </div>
                    
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <div>
                            <label for="pay_period_start" class="block text-sm font-medium text-gray-700 mb-2">
                                <i class="fas fa-calendar-plus mr-2 text-indigo-500"></i>Start Date *
                            </label>
                            <input type="date" 
                                   name="pay_period_start" 
                                   id="pay_period_start"
                                   value="{{ old('pay_period_start', $defaultPayPeriodStart->format('Y-m-d')) }}"
                                   class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent transition duration-200 @error('pay_period_start') border-red-500 @enderror"
                                   required>
                            @error('pay_period_start')
                            <p class="mt-2 text-sm text-red-600 flex items-center">
                                <i class="fas fa-exclamation-circle mr-2"></i> {{ $message }}
                            </p>
                            @enderror
                        </div>

                        <div>
                            <label for="pay_period_end" class="block text-sm font-medium text-gray-700 mb-2">
                                <i class="fas fa-calendar-minus mr-2 text-indigo-500"></i>End Date *
                            </label>
                            <input type="date" 
                                   name="pay_period_end" 
                                   id="pay_period_end"
                                   value="{{ old('pay_period_end', $defaultPayPeriodEnd->format('Y-m-d')) }}"
                                   class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent transition duration-200 @error('pay_period_end') border-red-500 @enderror"
                                   required>
                            @error('pay_period_end')
                            <p class="mt-2 text-sm text-red-600 flex items-center">
                                <i class="fas fa-exclamation-circle mr-2"></i> {{ $message }}
                            </p>
                            @enderror
                        </div>
                    </div>
                </div>

                <!-- Salary Details -->
                <div class="mb-8">
                    <div class="flex items-center mb-6">
                        <div class="w-8 h-8 rounded-lg bg-emerald-100 text-emerald-600 flex items-center justify-center mr-3">
                            <i class="fas fa-money-bill-wave"></i>
                        </div>
                        <h3 class="text-lg font-semibold text-gray-900">Salary Details</h3>
                    </div>
                    
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <!-- Basic Salary -->
                        <div>
                            <label for="basic_salary" class="block text-sm font-medium text-gray-700 mb-2">
                                <i class="fas fa-money-check mr-2 text-emerald-500"></i>Basic Salary *
                            </label>
                            <div class="relative">
                                <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                    <span class="text-gray-500 font-medium">EGP</span>
                                </div>
                                <input type="number" 
                                       name="basic_salary" 
                                       id="basic_salary"
                                       value="{{ old('basic_salary', 0) }}"
                                       class="w-full px-4 py-3 pl-16 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent transition duration-200 @error('basic_salary') border-red-500 @enderror"
                                       step="0.01"
                                       min="0"
                                       required>
                            </div>
                            @error('basic_salary')
                            <p class="mt-2 text-sm text-red-600 flex items-center">
                                <i class="fas fa-exclamation-circle mr-2"></i> {{ $message }}
                            </p>
                            @enderror
                        </div>

                        <!-- Allowances -->
                        <div>
                            <label for="allowances" class="block text-sm font-medium text-gray-700 mb-2">
                                <i class="fas fa-plus-circle mr-2 text-emerald-500"></i>Allowances
                            </label>
                            <div class="relative">
                                <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                    <span class="text-gray-500 font-medium">EGP</span>
                                </div>
                                <input type="number" 
                                       name="allowances" 
                                       id="allowances"
                                       value="{{ old('allowances', 0) }}"
                                       class="w-full px-4 py-3 pl-16 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent transition duration-200 @error('allowances') border-red-500 @enderror"
                                       step="0.01"
                                       min="0">
                            </div>
                            @error('allowances')
                            <p class="mt-2 text-sm text-red-600 flex items-center">
                                <i class="fas fa-exclamation-circle mr-2"></i> {{ $message }}
                            </p>
                            @enderror
                        </div>

                        <!-- Deductions -->
                        <div>
                            <label for="deductions" class="block text-sm font-medium text-gray-700 mb-2">
                                <i class="fas fa-minus-circle mr-2 text-red-500"></i>Deductions
                            </label>
                            <div class="relative">
                                <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                    <span class="text-gray-500 font-medium">EGP</span>
                                </div>
                                <input type="number" 
                                       name="deductions" 
                                       id="deductions"
                                       value="{{ old('deductions', 0) }}"
                                       class="w-full px-4 py-3 pl-16 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent transition duration-200 @error('deductions') border-red-500 @enderror"
                                       step="0.01"
                                       min="0">
                            </div>
                            @error('deductions')
                            <p class="mt-2 text-sm text-red-600 flex items-center">
                                <i class="fas fa-exclamation-circle mr-2"></i> {{ $message }}
                            </p>
                            @enderror
                        </div>

                        <!-- Overtime -->
                        <div>
                            <label for="overtime" class="block text-sm font-medium text-gray-700 mb-2">
                                <i class="fas fa-clock mr-2 text-amber-500"></i>Overtime
                            </label>
                            <div class="relative">
                                <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                    <span class="text-gray-500 font-medium">EGP</span>
                                </div>
                                <input type="number" 
                                       name="overtime" 
                                       id="overtime"
                                       value="{{ old('overtime', 0) }}"
                                       class="w-full px-4 py-3 pl-16 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent transition duration-200 @error('overtime') border-red-500 @enderror"
                                       step="0.01"
                                       min="0">
                            </div>
                            @error('overtime')
                            <p class="mt-2 text-sm text-red-600 flex items-center">
                                <i class="fas fa-exclamation-circle mr-2"></i> {{ $message }}
                            </p>
                            @enderror
                        </div>

                        <!-- Bonus -->
                        <div>
                            <label for="bonus" class="block text-sm font-medium text-gray-700 mb-2">
                                <i class="fas fa-gift mr-2 text-purple-500"></i>Bonus
                            </label>
                            <div class="relative">
                                <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                    <span class="text-gray-500 font-medium">EGP</span>
                                </div>
                                <input type="number" 
                                       name="bonus" 
                                       id="bonus"
                                       value="{{ old('bonus', 0) }}"
                                       class="w-full px-4 py-3 pl-16 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent transition duration-200 @error('bonus') border-red-500 @enderror"
                                       step="0.01"
                                       min="0">
                            </div>
                            @error('bonus')
                            <p class="mt-2 text-sm text-red-600 flex items-center">
                                <i class="fas fa-exclamation-circle mr-2"></i> {{ $message }}
                            </p>
                            @enderror
                        </div>
                    </div>

                    <!-- Net Salary Preview -->
                    <div class="mt-8 p-5 bg-gradient-to-r from-emerald-50 to-teal-50 rounded-xl border border-emerald-100">
                        <div class="flex flex-col md:flex-row md:items-center md:justify-between">
                            <div>
                                <p class="text-sm font-medium text-gray-600 mb-1">Net Salary Preview</p>
                                <p class="text-3xl font-bold text-emerald-600" id="netSalaryPreview">0.00 EGP</p>
                                <p class="text-xs text-gray-500 mt-2">
                                    Calculated: Basic + Allowances + Overtime + Bonus - Deductions
                                </p>
                            </div>
                            <div class="mt-4 md:mt-0">
                                <div class="grid grid-cols-2 gap-3">
                                    <div class="text-center p-3 bg-white rounded-lg border border-gray-200">
                                        <p class="text-xs text-gray-500">Total Income</p>
                                        <p class="text-sm font-semibold text-emerald-600" id="totalIncome">0.00</p>
                                    </div>
                                    <div class="text-center p-3 bg-white rounded-lg border border-gray-200">
                                        <p class="text-xs text-gray-500">Total Deductions</p>
                                        <p class="text-sm font-semibold text-red-600" id="totalDeductions">0.00</p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Payment Information -->
                <div class="mb-8">
                    <div class="flex items-center mb-6">
                        <div class="w-8 h-8 rounded-lg bg-purple-100 text-purple-600 flex items-center justify-center mr-3">
                            <i class="fas fa-calendar-check"></i>
                        </div>
                        <h3 class="text-lg font-semibold text-gray-900">Payment Information</h3>
                    </div>
                    
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <!-- Payment Date -->
                        <div>
                            <label for="payment_date" class="block text-sm font-medium text-gray-700 mb-2">
                                <i class="fas fa-calendar-day mr-2 text-purple-500"></i>Payment Date
                            </label>
                            <input type="date" 
                                   name="payment_date" 
                                   id="payment_date"
                                   value="{{ old('payment_date') }}"
                                   class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent transition duration-200 @error('payment_date') border-red-500 @enderror">
                            @error('payment_date')
                            <p class="mt-2 text-sm text-red-600 flex items-center">
                                <i class="fas fa-exclamation-circle mr-2"></i> {{ $message }}
                            </p>
                            @enderror
                        </div>
                        
                        <!-- Status (hidden, default is draft) -->
                        <input type="hidden" name="status" value="draft">
                        
                        <div class="p-4 bg-gray-50 rounded-lg border border-gray-200">
                            <div class="flex items-center">
                                <div class="w-8 h-8 rounded-lg bg-amber-100 text-amber-600 flex items-center justify-center mr-3">
                                    <i class="fas fa-clock"></i>
                                </div>
                                <div>
                                    <p class="text-sm font-medium text-gray-900">Status</p>
                                    <p class="text-xs text-gray-600">Payroll will be created as <span class="font-semibold text-amber-600">Draft</span></p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Notes -->
                <div class="mb-8">
                    <div class="flex items-center mb-4">
                        <div class="w-8 h-8 rounded-lg bg-gray-100 text-gray-600 flex items-center justify-center mr-3">
                            <i class="fas fa-sticky-note"></i>
                        </div>
                        <h3 class="text-lg font-semibold text-gray-900">Additional Notes</h3>
                    </div>
                    
                    <div>
                        <label for="notes" class="block text-sm font-medium text-gray-700 mb-2">
                            <i class="fas fa-comment-alt mr-2 text-gray-500"></i>Notes (Optional)
                        </label>
                        <textarea name="notes" 
                                  id="notes"
                                  rows="4"
                                  class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent transition duration-200 @error('notes') border-red-500 @enderror"
                                  placeholder="Add any notes or comments about this payroll...">{{ old('notes') }}</textarea>
                        @error('notes')
                        <p class="mt-2 text-sm text-red-600 flex items-center">
                            <i class="fas fa-exclamation-circle mr-2"></i> {{ $message }}
                        </p>
                        @enderror
                    </div>
                </div>
            </form>

            <!-- Form Actions -->
            <div class="px-6 py-5 border-t border-gray-200 bg-gray-50">
                <div class="flex flex-col md:flex-row md:items-center md:justify-between">
                    <div class="mb-4 md:mb-0">
                        <p class="text-sm text-gray-500 flex items-center">
                            <i class="fas fa-info-circle mr-2 text-blue-500"></i> Fields marked with * are required
                        </p>
                    </div>
                    <div class="flex items-center space-x-3">
                        <a href="{{ route('payrolls.index') }}" 
                           class="px-5 py-3 border border-gray-300 text-gray-700 font-medium rounded-lg hover:bg-gray-50 transition duration-200 flex items-center">
                            <i class="fas fa-times mr-2"></i> Cancel
                        </a>
                        <button type="submit" 
                                form="payrollForm"
                                class="px-5 py-3 bg-blue-600 hover:bg-blue-700 text-white font-medium rounded-lg transition duration-200 shadow-sm hover:shadow-md flex items-center group">
                            <i class="fas fa-save mr-2 group-hover:scale-110 transition-transform duration-200"></i> 
                            <span>Create Payroll</span>
                        </button>
                    </div>
                </div>
            </div>
        </div>

        <!-- Calculation Guide -->
        <div class="mt-8 bg-gradient-to-r from-blue-50 to-indigo-50 border border-blue-100 rounded-xl p-5">
            <div class="flex">
                <div class="flex-shrink-0">
                    <div class="w-10 h-10 rounded-lg bg-blue-100 text-blue-600 flex items-center justify-center">
                        <i class="fas fa-calculator"></i>
                    </div>
                </div>
                <div class="ml-4">
                    <h3 class="text-sm font-semibold text-blue-800">Payroll Calculation Guide</h3>
                    <div class="mt-2">
                        <ul class="space-y-2">
                            <li class="text-sm text-blue-700 flex items-start">
                                <i class="fas fa-check-circle text-blue-500 mr-2 mt-0.5 text-xs"></i>
                                <span><strong>Net Salary = (Basic + Allowances + Overtime + Bonus) - Deductions</strong></span>
                            </li>
                            <li class="text-sm text-blue-700 flex items-start">
                                <i class="fas fa-check-circle text-blue-500 mr-2 mt-0.5 text-xs"></i>
                                <span>Allowances include travel, housing, and other benefits</span>
                            </li>
                            <li class="text-sm text-blue-700 flex items-start">
                                <i class="fas fa-check-circle text-blue-500 mr-2 mt-0.5 text-xs"></i>
                                <span>Deductions include taxes, loans, and other withholdings</span>
                            </li>
                            <li class="text-sm text-blue-700 flex items-start">
                                <i class="fas fa-check-circle text-blue-500 mr-2 mt-0.5 text-xs"></i>
                                <span>Overtime is calculated based on company policy</span>
                            </li>
                        </ul>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
// Calculate salary totals
function calculateSalary() {
    const basic = parseFloat(document.getElementById('basic_salary').value) || 0;
    const allowances = parseFloat(document.getElementById('allowances').value) || 0;
    const deductions = parseFloat(document.getElementById('deductions').value) || 0;
    const overtime = parseFloat(document.getElementById('overtime').value) || 0;
    const bonus = parseFloat(document.getElementById('bonus').value) || 0;
    
    const totalIncome = basic + allowances + overtime + bonus;
    const netSalary = totalIncome - deductions;
    
    // Update displays
    document.getElementById('netSalaryPreview').textContent = 
        netSalary.toLocaleString('en-US', { minimumFractionDigits: 2, maximumFractionDigits: 2 }) + ' EGP';
    
    document.getElementById('totalIncome').textContent = 
        totalIncome.toLocaleString('en-US', { minimumFractionDigits: 2, maximumFractionDigits: 2 });
    
    document.getElementById('totalDeductions').textContent = 
        deductions.toLocaleString('en-US', { minimumFractionDigits: 2, maximumFractionDigits: 2 });
    
    return { netSalary, totalIncome, deductions };
}

// Fetch employee salary information
function fetchEmployeeSalary() {
    const employeeId = document.getElementById('employee_id').value;
    
    if (!employeeId) {
        document.getElementById('employeeInfo').classList.add('hidden');
        return;
    }
    
    // Show loading state
    document.getElementById('employeeInfo').classList.remove('hidden');
    document.getElementById('employeeDept').textContent = 'Loading...';
    document.getElementById('employeeSalary').textContent = 'Loading...';
    document.getElementById('employeeCurrency').textContent = 'Loading...';
    
    fetch(`/payrolls/employee-salary?employee_id=${employeeId}`, {
        headers: {
            'X-Requested-With': 'XMLHttpRequest',
            'X-CSRF-TOKEN': '{{ csrf_token() }}'
        }
    })
    .then(response => {
        if (!response.ok) {
            throw new Error('Employee not found');
        }
        return response.json();
    })
    .then(data => {
        document.getElementById('employeeDept').textContent = data.department || 'Not assigned';
        document.getElementById('employeeSalary').textContent = 
            data.basic_salary.toLocaleString('en-US', { minimumFractionDigits: 2, maximumFractionDigits: 2 }) + ' ' + data.currency;
        document.getElementById('employeeCurrency').textContent = data.currency;
        
        // Set the basic salary field with employee's salary
        document.getElementById('basic_salary').value = data.basic_salary;
        
        // Recalculate net salary
        calculateSalary();
    })
    .catch(error => {
        console.error('Error fetching employee salary:', error);
        document.getElementById('employeeInfo').classList.add('hidden');
        alert('Error loading employee information. Please try again.');
    });
}

// Add event listeners for salary calculation
document.addEventListener('DOMContentLoaded', function() {
    // Calculate initial salary
    calculateSalary();
    
    // Add event listeners to salary inputs
    const salaryInputs = ['basic_salary', 'allowances', 'deductions', 'overtime', 'bonus'];
    salaryInputs.forEach(id => {
        document.getElementById(id).addEventListener('input', calculateSalary);
    });
    
    // Validate form before submission
    document.getElementById('payrollForm').addEventListener('submit', function(e) {
        const { netSalary } = calculateSalary();
        
        // Validate net salary
        if (netSalary < 0) {
            e.preventDefault();
            alert('Net salary cannot be negative. Please check your deductions.');
            return false;
        }
        
        // Validate pay period
        const startDate = new Date(document.getElementById('pay_period_start').value);
        const endDate = new Date(document.getElementById('pay_period_end').value);
        
        if (endDate < startDate) {
            e.preventDefault();
            alert('Pay period end date must be after start date.');
            return false;
        }
        
        // Validate payment date if provided
        const paymentDate = document.getElementById('payment_date').value;
        if (paymentDate && new Date(paymentDate) < startDate) {
            e.preventDefault();
            alert('Payment date cannot be before pay period start date.');
            return false;
        }
        
        // Show loading state
        const submitBtn = this.querySelector('button[type="submit"]');
        const originalText = submitBtn.innerHTML;
        submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin mr-2"></i> Creating...';
        submitBtn.disabled = true;
        
        setTimeout(() => {
            submitBtn.innerHTML = originalText;
            submitBtn.disabled = false;
        }, 3000);
        
        return true;
    });
    
    // Auto-fetch employee salary if employee is pre-selected
    @if(old('employee_id'))
        fetchEmployeeSalary();
    @endif
});
</script>

<style>
/* Custom styles for form inputs */
input[type="number"]::-webkit-inner-spin-button,
input[type="number"]::-webkit-outer-spin-button {
    opacity: 1;
    margin: 0;
}

/* Focus styles */
input:focus, select:focus, textarea:focus {
    box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
}

/* Currency input styling */
.relative input {
    padding-left: 4rem;
}

/* Smooth transitions */
* {
    transition-property: background-color, border-color, color, fill, stroke, opacity, box-shadow, transform;
    transition-timing-function: cubic-bezier(0.4, 0, 0.2, 1);
    transition-duration: 200ms;
}

/* Form section styling */
.mb-8 {
    padding-bottom: 1.5rem;
    border-bottom: 1px solid #e5e7eb;
}

.mb-8:last-of-type {
    border-bottom: none;
}
</style>
@endsection