<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\Log;

class LoginController extends Controller
{
    /**
     * Show the login form.
     *
     * @return \Illuminate\View\View
     */
    public function showLoginForm()
    {
        // If user is already logged in, redirect based on role
        if (Auth::check()) {
            return $this->redirectBasedOnRole();
        }
        
        return view('auth.login');
    }

    /**
     * Handle a login request.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     *
     * @throws \Illuminate\Validation\ValidationException
     */
    public function login(Request $request)
    {
        // Validate the request
        $request->validate([
            'email' => 'required|email',
            'password' => 'required|string',
        ]);

        // Attempt to authenticate
        if (Auth::attempt($request->only('email', 'password'), $request->filled('remember'))) {
            $request->session()->regenerate();
            
            $user = Auth::user();
            Log::info('User logged in', ['user_id' => $user->id, 'email' => $user->email, 'roles' => $user->getRoleNames()]);
            
            // Check if user is active
            if ($user->status !== 'active') {
                Auth::logout();
                $request->session()->invalidate();
                $request->session()->regenerateToken();
                
                throw ValidationException::withMessages([
                    'email' => 'Your account is inactive. Please contact administrator.',
                ]);
            }
            
            // Redirect based on user role
            return $this->redirectBasedOnRole();
        }

        // Authentication failed
        throw ValidationException::withMessages([
            'email' => __('auth.failed'),
        ]);
    }

    /**
     * Log the user out of the application.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function logout(Request $request)
    {
        $user = Auth::user();
        Log::info('User logged out', ['user_id' => $user->id, 'email' => $user->email]);
        
        Auth::logout();
        
        $request->session()->invalidate();
        
        $request->session()->regenerateToken();
        
        return redirect('/login');
    }

    /**
     * Redirect user based on their role.
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    private function redirectBasedOnRole()
    {
        $user = Auth::user();
        
        // Check roles in priority order
        if ($user->hasRole('admin')) {
            Log::info('Redirecting admin to dashboard');
            return redirect()->route('dashboard')->with('success', 'Welcome back, Administrator!');
        }
        
        if ($user->hasRole('supervisor')) {
            Log::info('Redirecting supervisor to dashboard');
            return redirect()->route('dashboard')->with('success', 'Welcome back, Supervisor!');
        }
        
        if ($user->hasRole('employee')) {
            Log::info('Redirecting employee to employee dashboard');
            return redirect()->route('employee.dashboard')->with('success', 'Welcome back, ' . $user->name . '!');
        }
        
        // Default redirect if no role is assigned
        Log::warning('User has no roles assigned', ['user_id' => $user->id]);
        return redirect()->route('dashboard')->with('warning', 'Please contact administrator to assign you a role.');
    }
}