// app/Http/Controllers/Employee/EmployeeDashboardController.php
<?php

namespace App\Http\Controllers\Employee;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\Visit;
use App\Models\Attendance;
use App\Models\LeaveRequest;
use App\Models\Payroll;
use Carbon\Carbon;

class EmployeeDashboardController extends Controller
{
    public function index()
    {
        $user = Auth::user();
        $today = Carbon::today();
        
        // Today's visits
        $todayVisits = Visit::where('employee_id', $user->id)
            ->whereDate('scheduled_date', $today)
            ->orderBy('scheduled_time')
            ->get();
        
        // Completed visits
        $completedVisits = Visit::where('employee_id', $user->id)
            ->where('status', 'completed')
            ->whereMonth('scheduled_date', $today->month)
            ->count();
        
        // Pending visits
        $pendingVisits = Visit::where('employee_id', $user->id)
            ->where('status', 'scheduled')
            ->whereDate('scheduled_date', '>=', $today)
            ->count();
        
        // Leave balance (example calculation)
        $leaveBalance = 21 - LeaveRequest::where('employee_id', $user->id)
            ->where('status', 'approved')
            ->whereYear('created_at', $today->year)
            ->sum('total_days');
        
        // Today's attendance
        $todayAttendance = Attendance::where('employee_id', $user->id)
            ->whereDate('date', $today)
            ->first();
        
        return view('employee.dashboard', compact(
            'todayVisits',
            'completedVisits',
            'pendingVisits',
            'leaveBalance',
            'todayAttendance'
        ));
    }
    
    public function checkIn(Request $request)
    {
        $request->validate([
            'latitude' => 'required|numeric',
            'longitude' => 'required|numeric',
        ]);
        
        $user = Auth::user();
        $today = Carbon::today();
        
        // Check if already checked in
        $attendance = Attendance::firstOrCreate(
            [
                'employee_id' => $user->id,
                'date' => $today,
            ],
            [
                'check_in' => Carbon::now(),
                'check_in_latitude' => $request->latitude,
                'check_in_longitude' => $request->longitude,
                'status' => 'present',
            ]
        );
        
        if ($attendance->check_in) {
            return redirect()->back()->with('error', 'Already checked in today.');
        }
        
        $attendance->update([
            'check_in' => Carbon::now(),
            'check_in_latitude' => $request->latitude,
            'check_in_longitude' => $request->longitude,
            'status' => 'present',
        ]);
        
        return redirect()->back()->with('success', 'Checked in successfully.');
    }
    
    public function checkOut(Request $request)
    {
        $request->validate([
            'latitude' => 'required|numeric',
            'longitude' => 'required|numeric',
        ]);
        
        $user = Auth::user();
        $today = Carbon::today();
        
        $attendance = Attendance::where('employee_id', $user->id)
            ->whereDate('date', $today)
            ->first();
        
        if (!$attendance || !$attendance->check_in) {
            return redirect()->back()->with('error', 'Please check in first.');
        }
        
        if ($attendance->check_out) {
            return redirect()->back()->with('error', 'Already checked out today.');
        }
        
        // Calculate work hours
        $checkIn = Carbon::parse($attendance->check_in);
        $checkOut = Carbon::now();
        $workHours = $checkOut->diffInHours($checkIn);
        
        $attendance->update([
            'check_out' => $checkOut,
            'check_out_latitude' => $request->latitude,
            'check_out_longitude' => $request->longitude,
            'work_hours' => $workHours,
        ]);
        
        return redirect()->back()->with('success', 'Checked out successfully.');
    }
}