<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Department;
use App\Models\Client;
use App\Models\Visit;
use App\Models\LeaveRequest;
use App\Models\Payroll;
use App\Models\Governorate;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ReportController extends Controller
{
    public function index()
    {
        return view('reports.index');
    }

    public function generate(Request $request)
    {
        $request->validate([
            'report_type' => 'required|in:users,visits,leaves,payroll,clients,performance,summary',
            'date_from' => 'nullable|date',
            'date_to' => 'nullable|date|after_or_equal:date_from',
            'department_id' => 'nullable|exists:departments,id',
            'employee_id' => 'nullable|exists:users,id',
            'governorate_id' => 'nullable|exists:governorates,id',
            'status' => 'nullable|string',
            'format' => 'nullable|in:html,pdf,excel',
        ]);

        $dateFrom = $request->date_from ? Carbon::parse($request->date_from) : Carbon::now()->startOfMonth();
        $dateTo = $request->date_to ? Carbon::parse($request->date_to) : Carbon::now()->endOfMonth();

        $reportData = [];

        switch ($request->report_type) {
            case 'users':
                $reportData = $this->generateUserReport($dateFrom, $dateTo, $request);
                break;
            
            case 'visits':
                $reportData = $this->generateVisitReport($dateFrom, $dateTo, $request);
                break;
            
            case 'leaves':
                $reportData = $this->generateLeaveReport($dateFrom, $dateTo, $request);
                break;
            
            case 'payroll':
                $reportData = $this->generatePayrollReport($dateFrom, $dateTo, $request);
                break;
            
            case 'clients':
                $reportData = $this->generateClientReport($dateFrom, $dateTo, $request);
                break;
            
            case 'performance':
                $reportData = $this->generatePerformanceReport($dateFrom, $dateTo, $request);
                break;
            
            case 'summary':
                $reportData = $this->generateSummaryReport($dateFrom, $dateTo, $request);
                break;
        }

        if ($request->format === 'pdf') {
            return $this->exportToPDF($reportData, $request->report_type);
        } elseif ($request->format === 'excel') {
            return $this->exportToExcel($reportData, $request->report_type, $request);
        }

        return view('reports.view', compact('reportData', 'request'));
    }

    private function generateUserReport($dateFrom, $dateTo, $request)
    {
        $query = User::with('department')
            ->whereBetween('created_at', [$dateFrom, $dateTo]);

        if ($request->department_id) {
            $query->where('department_id', $request->department_id);
        }

        if ($request->status) {
            $query->where('status', $request->status);
        }

        $users = $query->get();

        $stats = [
            'total_users' => $users->count(),
            'active_users' => $users->where('status', 'active')->count(),
            'inactive_users' => $users->where('status', 'inactive')->count(),
            'by_department' => $users->groupBy('department.name')->map->count(),
            'by_role' => $users->groupBy('role')->map->count(),
            'new_users_this_month' => User::whereMonth('created_at', Carbon::now()->month)->count(),
        ];

        return [
            'type' => 'users',
            'title' => 'User Management Report',
            'period' => $dateFrom->format('M d, Y') . ' - ' . $dateTo->format('M d, Y'),
            'data' => $users,
            'stats' => $stats,
            'date_from' => $dateFrom,
            'date_to' => $dateTo,
        ];
    }

    private function generateVisitReport($dateFrom, $dateTo, $request)
    {
        $query = Visit::with(['employee', 'client', 'governorate', 'city'])
            ->whereBetween('scheduled_date', [$dateFrom, $dateTo]);

        if ($request->employee_id) {
            $query->where('employee_id', $request->employee_id);
        }

        if ($request->status) {
            $query->where('status', $request->status);
        }

        if ($request->governorate_id) {
            $query->where('governorate_id', $request->governorate_id);
        }

        $visits = $query->get();

        $stats = [
            'total_visits' => $visits->count(),
            'completed_visits' => $visits->where('status', 'completed')->count(),
            'scheduled_visits' => $visits->where('status', 'scheduled')->count(),
            'in_progress_visits' => $visits->where('status', 'in_progress')->count(),
            'cancelled_visits' => $visits->where('status', 'cancelled')->count(),
            'by_employee' => $visits->groupBy('employee.name')->map->count(),
            'by_governorate' => $visits->groupBy('governorate.name')->map->count(),
            'avg_completion_time' => $this->calculateAverageVisitCompletion($visits),
        ];

        return [
            'type' => 'visits',
            'title' => 'Visit Management Report',
            'period' => $dateFrom->format('M d, Y') . ' - ' . $dateTo->format('M d, Y'),
            'data' => $visits,
            'stats' => $stats,
            'date_from' => $dateFrom,
            'date_to' => $dateTo,
        ];
    }

    private function generateLeaveReport($dateFrom, $dateTo, $request)
    {
        $query = LeaveRequest::with(['employee', 'approver'])
            ->whereBetween('created_at', [$dateFrom, $dateTo]);

        if ($request->employee_id) {
            $query->where('employee_id', $request->employee_id);
        }

        if ($request->status) {
            $query->where('status', $request->status);
        }

        if ($request->department_id) {
            $query->whereHas('employee', function ($q) use ($request) {
                $q->where('department_id', $request->department_id);
            });
        }

        $leaves = $query->get();

        $stats = [
            'total_leaves' => $leaves->count(),
            'approved_leaves' => $leaves->where('status', 'approved')->count(),
            'pending_leaves' => $leaves->where('status', 'pending')->count(),
            'rejected_leaves' => $leaves->where('status', 'rejected')->count(),
            'by_type' => $leaves->groupBy('type')->map->count(),
            'by_employee' => $leaves->groupBy('employee.name')->map->count(),
            'total_days' => $leaves->sum('total_days'),
            'avg_days_per_leave' => $leaves->avg('total_days'),
        ];

        return [
            'type' => 'leaves',
            'title' => 'Leave Management Report',
            'period' => $dateFrom->format('M d, Y') . ' - ' . $dateTo->format('M d, Y'),
            'data' => $leaves,
            'stats' => $stats,
            'date_from' => $dateFrom,
            'date_to' => $dateTo,
        ];
    }

    private function generatePayrollReport($dateFrom, $dateTo, $request)
    {
        $query = Payroll::with(['employee'])
            ->whereBetween('pay_period_start', [$dateFrom, $dateTo]);

        if ($request->employee_id) {
            $query->where('employee_id', $request->employee_id);
        }

        if ($request->status) {
            $query->where('status', $request->status);
        }

        if ($request->department_id) {
            $query->whereHas('employee', function ($q) use ($request) {
                $q->where('department_id', $request->department_id);
            });
        }

        $payrolls = $query->get();

        $stats = [
            'total_payrolls' => $payrolls->count(),
            'draft_payrolls' => $payrolls->where('status', 'draft')->count(),
            'processed_payrolls' => $payrolls->where('status', 'processed')->count(),
            'paid_payrolls' => $payrolls->where('status', 'paid')->count(),
            'total_salary_paid' => $payrolls->where('status', 'paid')->sum('net_salary'),
            'avg_salary' => $payrolls->avg('net_salary'),
            'by_employee' => $payrolls->groupBy('employee.name')->map(function($items) {
                return [
                    'count' => $items->count(),
                    'total' => $items->sum('net_salary'),
                ];
            }),
            'by_month' => $payrolls->groupBy(function($item) {
                return $item->pay_period_start->format('Y-m');
            })->map->sum('net_salary'),
        ];

        return [
            'type' => 'payroll',
            'title' => 'Payroll Management Report',
            'period' => $dateFrom->format('M d, Y') . ' - ' . $dateTo->format('M d, Y'),
            'data' => $payrolls,
            'stats' => $stats,
            'date_from' => $dateFrom,
            'date_to' => $dateTo,
        ];
    }

    private function generateClientReport($dateFrom, $dateTo, $request)
    {
        $query = Client::with(['governorate', 'city'])
            ->whereBetween('created_at', [$dateFrom, $dateTo]);

        if ($request->governorate_id) {
            $query->where('governorate_id', $request->governorate_id);
        }

        if ($request->status) {
            $query->where('status', $request->status);
        }

        $clients = $query->get();

        $stats = [
            'total_clients' => $clients->count(),
            'active_clients' => $clients->where('status', 'active')->count(),
            'inactive_clients' => $clients->where('status', 'inactive')->count(),
            'by_governorate' => $clients->groupBy('governorate.name')->map->count(),
            'by_city' => $clients->groupBy('city.name')->map->count(),
            'by_type' => $clients->groupBy('type')->map->count(),
            'clients_with_visits' => $clients->filter(function($client) {
                return $client->visits()->count() > 0;
            })->count(),
        ];

        return [
            'type' => 'clients',
            'title' => 'Client Management Report',
            'period' => $dateFrom->format('M d, Y') . ' - ' . $dateTo->format('M d, Y'),
            'data' => $clients,
            'stats' => $stats,
            'date_from' => $dateFrom,
            'date_to' => $dateTo,
        ];
    }

    private function generatePerformanceReport($dateFrom, $dateTo, $request)
    {
        // Employee Performance Metrics
        $query = User::role('employee')->with(['department']);
        
        if ($request->department_id) {
            $query->where('department_id', $request->department_id);
        }
        
        $employees = $query->get();
        
        $performanceData = [];
        
        foreach ($employees as $employee) {
            $visits = Visit::where('employee_id', $employee->id)
                ->whereBetween('scheduled_date', [$dateFrom, $dateTo])
                ->get();
            
            $completedVisits = $visits->where('status', 'completed')->count();
            $totalVisits = $visits->count();
            
            $leaves = LeaveRequest::where('employee_id', $employee->id)
                ->whereBetween('created_at', [$dateFrom, $dateTo])
                ->where('status', 'approved')
                ->get();
            
            $totalLeaveDays = $leaves->sum('total_days');
            
            $performanceData[] = [
                'employee' => $employee,
                'total_visits' => $totalVisits,
                'completed_visits' => $completedVisits,
                'completion_rate' => $totalVisits > 0 ? ($completedVisits / $totalVisits) * 100 : 0,
                'total_leave_days' => $totalLeaveDays,
                'department' => $employee->department->name ?? 'N/A',
                'avg_visit_duration' => $this->calculateAverageVisitDuration($employee->id, $dateFrom, $dateTo),
            ];
        }

        $stats = [
            'total_employees' => $employees->count(),
            'avg_completion_rate' => collect($performanceData)->avg('completion_rate'),
            'top_performers' => collect($performanceData)->sortByDesc('completion_rate')->take(5)->values(),
            'most_leave_days' => collect($performanceData)->sortByDesc('total_leave_days')->take(5)->values(),
            'by_department' => collect($performanceData)->groupBy('department')->map(function($items) {
                return [
                    'count' => $items->count(),
                    'avg_completion_rate' => $items->avg('completion_rate'),
                ];
            }),
        ];

        return [
            'type' => 'performance',
            'title' => 'Employee Performance Report',
            'period' => $dateFrom->format('M d, Y') . ' - ' . $dateTo->format('M d, Y'),
            'data' => $performanceData,
            'stats' => $stats,
            'date_from' => $dateFrom,
            'date_to' => $dateTo,
        ];
    }

    private function generateSummaryReport($dateFrom, $dateTo, $request)
    {
        // Overall System Summary
        $summary = [
            'users' => [
                'total' => User::count(),
                'active' => User::where('status', 'active')->count(),
                'employees' => User::role('employee')->count(),
                'admins' => User::role('admin')->count(),
                'supervisors' => User::role('supervisor')->count(),
            ],
            'visits' => [
                'total' => Visit::whereBetween('scheduled_date', [$dateFrom, $dateTo])->count(),
                'completed' => Visit::where('status', 'completed')->whereBetween('scheduled_date', [$dateFrom, $dateTo])->count(),
                'in_progress' => Visit::where('status', 'in_progress')->count(),
                'scheduled' => Visit::where('status', 'scheduled')->whereBetween('scheduled_date', [$dateFrom, $dateTo])->count(),
            ],
            'leaves' => [
                'total' => LeaveRequest::whereBetween('created_at', [$dateFrom, $dateTo])->count(),
                'approved' => LeaveRequest::where('status', 'approved')->whereBetween('created_at', [$dateFrom, $dateTo])->count(),
                'pending' => LeaveRequest::where('status', 'pending')->count(),
                'total_days' => LeaveRequest::where('status', 'approved')->whereBetween('created_at', [$dateFrom, $dateTo])->sum('total_days'),
            ],
            'payroll' => [
                'total' => Payroll::whereBetween('pay_period_start', [$dateFrom, $dateTo])->count(),
                'paid' => Payroll::where('status', 'paid')->whereBetween('pay_period_start', [$dateFrom, $dateTo])->count(),
                'total_amount' => Payroll::where('status', 'paid')->whereBetween('pay_period_start', [$dateFrom, $dateTo])->sum('net_salary'),
            ],
            'clients' => [
                'total' => Client::count(),
                'active' => Client::where('status', 'active')->count(),
                'by_governorate' => Client::groupBy('governorate_id')->selectRaw('count(*) as count, governorate_id')->get(),
            ],
        ];

        // Monthly trends
        $monthlyTrends = [];
        for ($i = 5; $i >= 0; $i--) {
            $month = Carbon::now()->subMonths($i);
            $start = $month->copy()->startOfMonth();
            $end = $month->copy()->endOfMonth();
            
            $monthlyTrends[$month->format('M Y')] = [
                'visits' => Visit::whereBetween('scheduled_date', [$start, $end])->count(),
                'leaves' => LeaveRequest::whereBetween('created_at', [$start, $end])->count(),
                'payroll' => Payroll::whereBetween('pay_period_start', [$start, $end])->count(),
            ];
        }

        return [
            'type' => 'summary',
            'title' => 'System Summary Report',
            'period' => $dateFrom->format('M d, Y') . ' - ' . $dateTo->format('M d, Y'),
            'summary' => $summary,
            'monthly_trends' => $monthlyTrends,
            'date_from' => $dateFrom,
            'date_to' => $dateTo,
        ];
    }

    private function calculateAverageVisitCompletion($visits)
    {
        $completedVisits = $visits->where('status', 'completed')
            ->whereNotNull('started_at')
            ->whereNotNull('completed_at');
        
        if ($completedVisits->isEmpty()) {
            return 0;
        }
        
        $totalMinutes = $completedVisits->sum(function($visit) {
            return $visit->started_at->diffInMinutes($visit->completed_at);
        });
        
        return round($totalMinutes / $completedVisits->count(), 2);
    }

    private function calculateAverageVisitDuration($employeeId, $dateFrom, $dateTo)
    {
        $visits = Visit::where('employee_id', $employeeId)
            ->where('status', 'completed')
            ->whereNotNull('started_at')
            ->whereNotNull('completed_at')
            ->whereBetween('scheduled_date', [$dateFrom, $dateTo])
            ->get();
        
        if ($visits->isEmpty()) {
            return 0;
        }
        
        $totalMinutes = $visits->sum(function($visit) {
            return $visit->started_at->diffInMinutes($visit->completed_at);
        });
        
        return round($totalMinutes / $visits->count(), 2);
    }

    private function exportToPDF($reportData, $reportType)
    {
        // Check if PDF package is installed
        if (!class_exists(\Barryvdh\DomPDF\Facade\Pdf::class)) {
            // Fallback to view with download instructions
            return view('reports.export-instructions', [
                'message' => 'PDF export requires dompdf package installation',
                'install_command' => 'composer require barryvdh/laravel-dompdf',
                'package' => 'pdf',
            ]);
        }
        
        try {
            $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('reports.pdf.' . $reportType, compact('reportData'));
            return $pdf->download('report-' . $reportType . '-' . now()->format('Y-m-d') . '.pdf');
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to generate PDF: ' . $e->getMessage());
        }
    }

    private function exportToExcel($reportData, $reportType, $request)
    {
        // Check if Excel package is installed
        if (!class_exists(\Maatwebsite\Excel\Facades\Excel::class)) {
            // Fallback to view with download instructions
            return view('reports.export-instructions', [
                'message' => 'Excel export requires maatwebsite/excel package installation',
                'install_command' => 'composer require maatwebsite/excel',
                'package' => 'excel',
            ]);
        }
        
        try {
            // Generate Excel file
            $fileName = 'report-' . $reportType . '-' . now()->format('Y-m-d-H-i') . '.xlsx';
            
            return \Maatwebsite\Excel\Facades\Excel::download(
                new \App\Exports\ReportExport($reportData, $reportType),
                $fileName
            );
            
        } catch (\Exception $e) {
            // If export fails, show error
            return back()->with('error', 'Failed to generate Excel file: ' . $e->getMessage());
        }
    }

    // API endpoints for dashboard widgets
    public function dashboardStats()
    {
        try {
            $today = Carbon::today();
            
            $stats = [
                'total_employees' => User::role('employee')->count(),
                'active_visits_today' => Visit::whereDate('scheduled_date', $today)
                    ->whereIn('status', ['scheduled', 'in_progress'])
                    ->count(),
                'pending_leaves' => LeaveRequest::where('status', 'pending')->count(),
                'total_clients' => Client::count(),
                'recent_payrolls' => Payroll::where('status', 'paid')
                    ->orderBy('payment_date', 'desc')
                    ->take(5)
                    ->get()
                    ->map(function($payroll) {
                        return [
                            'employee' => ['name' => $payroll->employee->name],
                            'pay_period_start' => $payroll->pay_period_start->format('M d'),
                            'net_salary' => number_format($payroll->net_salary, 2),
                        ];
                    }),
                'monthly_visit_trend' => $this->getMonthlyVisitTrend(),
            ];
            
            return response()->json($stats);
        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Failed to fetch dashboard stats',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    private function getMonthlyVisitTrend()
    {
        $data = [];
        for ($i = 5; $i >= 0; $i--) {
            $month = Carbon::now()->subMonths($i);
            $start = $month->copy()->startOfMonth();
            $end = $month->copy()->endOfMonth();
            
            $data[] = [
                'month' => $month->format('M'),
                'visits' => Visit::whereBetween('scheduled_date', [$start, $end])->count(),
                'completed' => Visit::where('status', 'completed')
                    ->whereBetween('scheduled_date', [$start, $end])
                    ->count(),
            ];
        }
        
        return $data;
    }

    // Additional filters for the report form
    public function getFilters($reportType)
    {
        $filters = [];
        
        switch ($reportType) {
            case 'users':
                $filters['departments'] = Department::all();
                $filters['statuses'] = ['active', 'inactive'];
                break;
                
            case 'visits':
                $filters['employees'] = User::role('employee')->get();
                $filters['departments'] = Department::all();
                $filters['governorates'] = Governorate::all();
                $filters['statuses'] = ['scheduled', 'in_progress', 'completed', 'cancelled'];
                break;
                
            case 'leaves':
                $filters['employees'] = User::role('employee')->get();
                $filters['departments'] = Department::all();
                $filters['statuses'] = ['pending', 'approved', 'rejected'];
                break;
                
            case 'payroll':
                $filters['employees'] = User::role('employee')->get();
                $filters['departments'] = Department::all();
                $filters['statuses'] = ['draft', 'processed', 'paid'];
                break;
                
            case 'clients':
                $filters['governorates'] = Governorate::all();
                $filters['statuses'] = ['active', 'inactive'];
                break;
                
            case 'performance':
                $filters['departments'] = Department::all();
                break;
        }
        
        return response()->json($filters);
    }
}