<?php
require __DIR__ . '/vendor/autoload.php';
$app = require_once __DIR__ . '/bootstrap/app.php';
$app->make(Illuminate\Contracts\Console\Kernel::class)->bootstrap();

echo "<h3>Fixing 419 Error and Map Issues...</h3>";

try {
    // 1. Clear all caches
    echo "1. Clearing caches...<br>";
    \Artisan::call('cache:clear');
    \Artisan::call('config:clear');
    \Artisan::call('view:clear');
    \Artisan::call('route:clear');
    \Artisan::call('optimize:clear');
    echo "✓ Caches cleared<br><br>";
    
    // 2. Fix permissions
    echo "2. Fixing permissions...<br>";
    chmod(__DIR__ . '/storage', 0755);
    chmod(__DIR__ . '/bootstrap/cache', 0755);
    echo "✓ Permissions fixed<br><br>";
    
    // 3. Switch to file sessions
    echo "3. Configuring sessions...<br>";
    $envPath = __DIR__ . '/.env';
    $envContent = file_get_contents($envPath);
    $envContent = str_replace('SESSION_DRIVER=database', 'SESSION_DRIVER=file', $envContent);
    $envContent = str_replace('SESSION_SECURE_COOKIE=false', 'SESSION_SECURE_COOKIE=true', $envContent);
    file_put_contents($envPath, $envContent);
    
    // Create sessions directory
    $sessionsDir = __DIR__ . '/storage/framework/sessions';
    if (!is_dir($sessionsDir)) {
        mkdir($sessionsDir, 0755, true);
    }
    echo "✓ Sessions configured (using file driver)<br><br>";
    
    // 4. Check CSRF token in login form
    echo "4. Checking login form...<br>";
    $loginPath = __DIR__ . '/resources/views/auth/login.blade.php';
    if (file_exists($loginPath)) {
        $loginContent = file_get_contents($loginPath);
        if (strpos($loginContent, '@csrf') !== false) {
            echo "✓ CSRF token found in login form<br><br>";
        } else {
            echo "⚠ CSRF token missing in login form - adding...<br>";
            $loginContent = str_replace(
                '<form method="POST" action="{{ route(\'login\') }}">',
                '<form method="POST" action="{{ route(\'login\') }}">' . PHP_EOL . '    @csrf',
                $loginContent
            );
            file_put_contents($loginPath, $loginContent);
            echo "✓ CSRF token added to login form<br><br>";
        }
    }
    
    // 5. Create a simple working map view
    echo "5. Creating working map view...<br>";
    $mapViewPath = __DIR__ . '/resources/views/live-tracking/map.blade.php';
    $simpleMap = <<<'EOD'
@extends('layouts.dashboard')

@section('title', 'Live Tracking Map')

@section('content')
<div class="container mx-auto px-4">
    <h1 class="text-2xl font-bold text-gray-800 mb-6">Live Tracking Map</h1>
    
    <div class="bg-white shadow rounded-lg p-4 mb-6">
        <div class="mb-4">
            <h2 class="text-lg font-semibold">Interactive Map</h2>
            <p class="text-gray-600">Real-time employee tracking</p>
        </div>
        
        <!-- Simple Map Container -->
        <div id="map" style="height: 500px; width: 100%; border: 1px solid #ddd; border-radius: 0.5rem;"></div>
        
        <div class="mt-4 flex space-x-3">
            <button onclick="centerMap()" class="px-4 py-2 bg-blue-600 text-white rounded hover:bg-blue-700">
                Center Map
            </button>
            <button onclick="addMarkers()" class="px-4 py-2 bg-green-600 text-white rounded hover:bg-green-700">
                Add Test Markers
            </button>
            <a href="{{ route('live-tracking.index') }}" class="px-4 py-2 bg-gray-600 text-white rounded hover:bg-gray-700">
                Back to Dashboard
            </a>
        </div>
    </div>
    
    <!-- Status -->
    <div class="bg-gray-50 p-4 rounded-lg">
        <h3 class="font-semibold mb-2">Status:</h3>
        <p id="status">Ready to load map...</p>
    </div>
</div>

<!-- Load Leaflet -->
<link rel="stylesheet" href="https://unpkg.com/leaflet@1.9.4/dist/leaflet.css" />
<script src="https://unpkg.com/leaflet@1.9.4/dist/leaflet.js"></script>

<script>
let map;
let markers = [];

document.addEventListener('DOMContentLoaded', function() {
    console.log('DOM loaded, creating map...');
    document.getElementById('status').textContent = 'Creating map...';
    
    try {
        // Create map centered on Cairo
        map = L.map('map').setView([30.0444, 31.2357], 13);
        
        // Add OpenStreetMap tiles
        L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
            attribution: '© OpenStreetMap contributors'
        }).addTo(map);
        
        // Add initial marker
        L.marker([30.0444, 31.2357])
            .addTo(map)
            .bindPopup('Map is working!<br>GPS Tracking System')
            .openPopup();
        
        console.log('Map created successfully!');
        document.getElementById('status').textContent = 'Map loaded successfully!';
        
    } catch (error) {
        console.error('Error:', error);
        document.getElementById('status').textContent = 'Error: ' + error.message;
        document.getElementById('status').style.color = 'red';
    }
});

function centerMap() {
    if (map) {
        map.setView([30.0444, 31.2357], 13);
        document.getElementById('status').textContent = 'Map centered at ' + new Date().toLocaleTimeString();
    }
}

function addMarkers() {
    if (!map) return;
    
    // Clear existing markers
    markers.forEach(marker => map.removeLayer(marker));
    markers = [];
    
    // Add test markers
    const testLocations = [
        [30.0444, 31.2357, 'Central Office'],
        [30.0489, 31.2386, 'Client A'],
        [30.0400, 31.2330, 'Client B'],
        [30.0550, 31.2450, 'Warehouse'],
        [30.0350, 31.2250, 'Branch Office']
    ];
    
    testLocations.forEach(([lat, lng, title]) => {
        const marker = L.marker([lat, lng])
            .addTo(map)
            .bindPopup(title);
        markers.push(marker);
    });
    
    document.getElementById('status').textContent = 'Added ' + markers.length + ' test markers';
}
</script>
@endsection
EOD;

    file_put_contents($mapViewPath, $simpleMap);
    echo "✓ Created working map view<br><br>";
    
    // 6. Clear config cache again
    \Artisan::call('config:clear');
    
    echo "<h4 style='color: green;'>✓ All fixes applied successfully!</h4>";
    echo "<p><strong>Next steps:</strong></p>";
    echo "<ol>";
    echo "<li><a href='/login' target='_blank'>Test Login Page</a> (should work now)</li>";
    echo "<li><a href='/live-tracking/map' target='_blank'>Test Map Page</a> (should show map)</li>";
    echo "<li>Check browser console (F12) for any JavaScript errors</li>";
    echo "</ol>";
    
} catch (Exception $e) {
    echo "<h4 style='color: red;'>Error:</h4>";
    echo "<pre>" . $e->getMessage() . "</pre>";
    echo "<pre>" . $e->getTraceAsString() . "</pre>";
}
?>
