// Innovaxcess Attendance - Service Worker
const CACHE_NAME = 'innovaxcess-v1';
const urlsToCache = [
    '/',
    '/dashboard',
    '/manifest.json',
    '/favicon.ico',
    // Add your CSS and JS files here
];

// Install event
self.addEventListener('install', event => {
    event.waitUntil(
        caches.open(CACHE_NAME)
            .then(cache => {
                console.log('Opened cache');
                return cache.addAll(urlsToCache);
            })
    );
});

// Activate event
self.addEventListener('activate', event => {
    event.waitUntil(
        caches.keys().then(cacheNames => {
            return Promise.all(
                cacheNames.map(cacheName => {
                    if (cacheName !== CACHE_NAME) {
                        console.log('Deleting old cache:', cacheName);
                        return caches.delete(cacheName);
                    }
                })
            );
        })
    );
});

// Fetch event
self.addEventListener('fetch', event => {
    event.respondWith(
        caches.match(event.request)
            .then(response => {
                // Cache hit - return response
                if (response) {
                    return response;
                }

                // Clone the request
                const fetchRequest = event.request.clone();

                return fetch(fetchRequest).then(response => {
                    // Check if valid response
                    if (!response || response.status !== 200 || response.type !== 'basic') {
                        return response;
                    }

                    // Clone the response
                    const responseToCache = response.clone();

                    caches.open(CACHE_NAME)
                        .then(cache => {
                            cache.put(event.request, responseToCache);
                        });

                    return response;
                });
            })
            .catch(() => {
                // If both cache and network fail, show offline page
                if (event.request.mode === 'navigate') {
                    return caches.match('/offline.html');
                }
            })
    );
});

// Background sync for offline data
self.addEventListener('sync', event => {
    if (event.tag === 'sync-visits') {
        event.waitUntil(syncVisits());
    }
    if (event.tag === 'sync-leaves') {
        event.waitUntil(syncLeaves());
    }
});

// Sync visits data
async function syncVisits() {
    try {
        const db = await openDB();
        const tx = db.transaction('pendingVisits', 'readwrite');
        const store = tx.objectStore('pendingVisits');
        const pendingVisits = await store.getAll();
        
        for (const visit of pendingVisits) {
            try {
                const response = await fetch('/api/visits', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': getCSRFToken()
                    },
                    body: JSON.stringify(visit)
                });
                
                if (response.ok) {
                    await store.delete(visit.id);
                    console.log('Synced visit:', visit.id);
                }
            } catch (error) {
                console.error('Failed to sync visit:', error);
            }
        }
        
        await tx.done;
    } catch (error) {
        console.error('Sync error:', error);
    }
}

// Sync leaves data
async function syncLeaves() {
    try {
        const db = await openDB();
        const tx = db.transaction('pendingLeaves', 'readwrite');
        const store = tx.objectStore('pendingLeaves');
        const pendingLeaves = await store.getAll();
        
        for (const leave of pendingLeaves) {
            try {
                const response = await fetch('/api/leaves', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': getCSRFToken()
                    },
                    body: JSON.stringify(leave)
                });
                
                if (response.ok) {
                    await store.delete(leave.id);
                    console.log('Synced leave:', leave.id);
                }
            } catch (error) {
                console.error('Failed to sync leave:', error);
            }
        }
        
        await tx.done;
    } catch (error) {
        console.error('Sync error:', error);
    }
}

// Push notifications
self.addEventListener('push', event => {
    if (!event.data) return;
    
    const data = event.data.json();
    const options = {
        body: data.body,
        icon: '/images/icon-192x192.png',
        badge: '/images/badge-72x72.png',
        vibrate: [100, 50, 100],
        data: {
            dateOfArrival: Date.now(),
            primaryKey: 1
        },
        actions: [
            {
                action: 'view',
                title: 'View'
            },
            {
                action: 'close',
                title: 'Close'
            }
        ]
    };
    
    event.waitUntil(
        self.registration.showNotification(data.title, options)
    );
});

// Notification click event
self.addEventListener('notificationclick', event => {
    event.notification.close();
    
    if (event.action === 'view') {
        event.waitUntil(
            clients.openWindow(event.notification.data.url || '/')
        );
    }
});

// Open IndexedDB
function openDB() {
    return new Promise((resolve, reject) => {
        const request = indexedDB.open('InnovaxcessDB', 1);
        
        request.onupgradeneeded = event => {
            const db = event.target.result;
            
            // Create object stores
            if (!db.objectStoreNames.contains('pendingVisits')) {
                db.createObjectStore('pendingVisits', { keyPath: 'id' });
            }
            if (!db.objectStoreNames.contains('pendingLeaves')) {
                db.createObjectStore('pendingLeaves', { keyPath: 'id' });
            }
            if (!db.objectStoreNames.contains('locationData')) {
                db.createObjectStore('locationData', { keyPath: 'timestamp' });
            }
        };
        
        request.onsuccess = event => resolve(event.target.result);
        request.onerror = event => reject(event.target.error);
    });
}

// Get CSRF token from meta tag
function getCSRFToken() {
    // This would be stored in IndexedDB when the app loads
    return localStorage.getItem('csrf_token') || '';
}