@extends('layouts.dashboard')

@section('title', 'Request Leave')

@section('content')
<div class="container mx-auto px-4 py-6">
    <!-- Header -->
    <div class="flex items-center justify-between mb-8">
        <div>
            <a href="{{ route('leaves.my-leaves') }}" 
               class="text-primary-600 hover:text-primary-700 transition-colors duration-200 inline-flex items-center mb-2">
                <i class="fas fa-arrow-left mr-2"></i> Back to My Leaves
            </a>
            <h1 class="text-2xl font-bold text-gray-800">Request Leave</h1>
            <p class="text-gray-600 mt-1">Submit a new leave request</p>
        </div>
        <div class="flex items-center space-x-2">
            <div class="w-12 h-12 rounded-lg bg-gradient-to-r from-blue-100 to-indigo-200 flex items-center justify-center">
                <i class="fas fa-calendar-plus text-blue-600 text-xl"></i>
            </div>
        </div>
    </div>

    <div class="max-w-2xl mx-auto">
        <!-- Leave Balance Card -->
        <div class="card-hover bg-gradient-to-r from-blue-500 to-blue-600 rounded-xl shadow-md p-6 mb-6 text-white">
            <div class="flex items-center justify-between">
                <div>
                    <h2 class="text-lg font-semibold mb-1">Your Leave Balance</h2>
                    <p class="opacity-90 text-sm">Annual leave entitlement</p>
                </div>
                <div class="text-right">
                    <div class="text-3xl font-bold">{{ $availableLeaves }} / {{ $totalLeaves }}</div>
                    <div class="text-sm opacity-90">days available</div>
                </div>
            </div>
            <div class="mt-4">
                <div class="h-2 bg-white/20 rounded-full overflow-hidden">
                    <div class="h-full bg-white rounded-full" 
                         style="width: {{ ($availableLeaves / $totalLeaves) * 100 }}%"></div>
                </div>
                <div class="flex justify-between text-xs mt-2">
                    <span>Used: {{ $totalLeaves - $availableLeaves }} days</span>
                    <span>Available: {{ $availableLeaves }} days</span>
                </div>
            </div>
        </div>

        <!-- Leave Request Form -->
        <div class="bg-white rounded-xl shadow-sm border border-gray-100 overflow-hidden">
            <!-- Form Header -->
            <div class="px-6 py-4 border-b border-gray-100 bg-gradient-to-r from-green-50 to-teal-100">
                <h2 class="text-lg font-semibold text-gray-800">Leave Request Form</h2>
                <p class="text-sm text-gray-600 mt-1">Fill in your leave details</p>
            </div>

            <!-- Form -->
            <form action="{{ route('leaves.store') }}" method="POST" id="leaveForm" class="p-6">
                @csrf

                <div class="space-y-6">
                    <!-- Leave Type -->
                    <div>
                        <label class="form-label">
                            <i class="fas fa-info-circle mr-2 text-gray-400"></i>Leave Type *
                        </label>
                        <div class="grid grid-cols-2 md:grid-cols-4 gap-3">
                            <label class="cursor-pointer">
                                <input type="radio" name="type" value="sick" class="sr-only peer" required>
                                <div class="p-4 border border-gray-200 rounded-lg text-center peer-checked:border-blue-500 peer-checked:bg-blue-50 hover:bg-gray-50 transition duration-200">
                                    <i class="fas fa-heartbeat text-red-500 text-xl mb-2"></i>
                                    <div class="font-medium text-gray-900">Sick Leave</div>
                                    <div class="text-xs text-gray-500">Medical reasons</div>
                                </div>
                            </label>
                            
                            <label class="cursor-pointer">
                                <input type="radio" name="type" value="vacation" class="sr-only peer">
                                <div class="p-4 border border-gray-200 rounded-lg text-center peer-checked:border-blue-500 peer-checked:bg-blue-50 hover:bg-gray-50 transition duration-200">
                                    <i class="fas fa-umbrella-beach text-blue-500 text-xl mb-2"></i>
                                    <div class="font-medium text-gray-900">Vacation</div>
                                    <div class="text-xs text-gray-500">Annual leave</div>
                                </div>
                            </label>
                            
                            <label class="cursor-pointer">
                                <input type="radio" name="type" value="personal" class="sr-only peer">
                                <div class="p-4 border border-gray-200 rounded-lg text-center peer-checked:border-blue-500 peer-checked:bg-blue-50 hover:bg-gray-50 transition duration-200">
                                    <i class="fas fa-user text-yellow-500 text-xl mb-2"></i>
                                    <div class="font-medium text-gray-900">Personal</div>
                                    <div class="text-xs text-gray-500">Personal matters</div>
                                </div>
                            </label>
                            
                            <label class="cursor-pointer">
                                <input type="radio" name="type" value="emergency" class="sr-only peer">
                                <div class="p-4 border border-gray-200 rounded-lg text-center peer-checked:border-blue-500 peer-checked:bg-blue-50 hover:bg-gray-50 transition duration-200">
                                    <i class="fas fa-exclamation-triangle text-purple-500 text-xl mb-2"></i>
                                    <div class="font-medium text-gray-900">Emergency</div>
                                    <div class="text-xs text-gray-500">Urgent situations</div>
                                </div>
                            </label>
                        </div>
                        @error('type')
                            <p class="mt-1 text-sm text-red-600 flex items-center">
                                <i class="fas fa-exclamation-circle mr-1"></i> {{ $message }}
                            </p>
                        @enderror
                    </div>

                    <!-- Leave Dates -->
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <!-- Start Date -->
                        <div>
                            <label for="start_date" class="form-label">
                                <i class="fas fa-calendar-day mr-2 text-gray-400"></i>Start Date *
                            </label>
                            <input type="date" 
                                   id="start_date" 
                                   name="start_date" 
                                   value="{{ old('start_date') }}"
                                   class="form-input @error('start_date') border-red-500 @enderror"
                                   required>
                            @error('start_date')
                                <p class="mt-1 text-sm text-red-600 flex items-center">
                                    <i class="fas fa-exclamation-circle mr-1"></i> {{ $message }}
                                </p>
                            @enderror
                        </div>

                        <!-- End Date -->
                        <div>
                            <label for="end_date" class="form-label">
                                <i class="fas fa-calendar-alt mr-2 text-gray-400"></i>End Date *
                            </label>
                            <input type="date" 
                                   id="end_date" 
                                   name="end_date" 
                                   value="{{ old('end_date') }}"
                                   class="form-input @error('end_date') border-red-500 @enderror"
                                   required>
                            @error('end_date')
                                <p class="mt-1 text-sm text-red-600 flex items-center">
                                    <i class="fas fa-exclamation-circle mr-1"></i> {{ $message }}
                                </p>
                            @enderror
                        </div>
                    </div>

                    <!-- Duration Preview -->
                    <div id="durationPreview" class="hidden p-4 bg-blue-50 border border-blue-100 rounded-lg">
                        <div class="flex items-center justify-between">
                            <div>
                                <div class="text-sm font-medium text-blue-800">Leave Duration</div>
                                <div class="text-sm text-blue-600" id="daysCount">0 days</div>
                            </div>
                            <div class="text-right">
                                <div class="text-sm text-blue-600" id="dateRange"></div>
                                <div class="text-xs text-blue-500" id="leaveBalance"></div>
                            </div>
                        </div>
                    </div>

                    <!-- Reason -->
                    <div>
                        <label for="reason" class="form-label">
                            <i class="fas fa-comment-alt mr-2 text-gray-400"></i>Reason for Leave *
                        </label>
                        <textarea id="reason" 
                                  name="reason" 
                                  rows="4"
                                  class="form-input @error('reason') border-red-500 @enderror"
                                  placeholder="Please provide details about why you need this leave..."
                                  required>{{ old('reason') }}</textarea>
                        @error('reason')
                            <p class="mt-1 text-sm text-red-600 flex items-center">
                                <i class="fas fa-exclamation-circle mr-1"></i> {{ $message }}
                            </p>
                        @enderror
                        <div class="flex justify-between items-center mt-1">
                            <p class="text-sm text-gray-500">Provide clear details for approval</p>
                            <span id="charCount" class="text-xs text-gray-400">0 characters</span>
                        </div>
                    </div>
                </div>

                <!-- Form Footer -->
                <div class="mt-8 pt-6 border-t border-gray-100 flex items-center justify-between">
                    <div>
                        <p class="text-sm text-gray-500">
                            <i class="fas fa-info-circle mr-1"></i> Fields marked with * are required
                        </p>
                    </div>
                    <div class="flex items-center space-x-3">
                        <a href="{{ route('leaves.my-leaves') }}" 
                           class="btn-secondary">
                            <i class="fas fa-times mr-2"></i> Cancel
                        </a>
                        <button type="submit" 
                                class="btn-primary">
                            <i class="fas fa-paper-plane mr-2"></i> Submit Request
                        </button>
                    </div>
                </div>
            </form>
        </div>

        <!-- Important Notes -->
        <div class="mt-6 bg-yellow-50 border border-yellow-100 rounded-lg p-4">
            <div class="flex">
                <div class="flex-shrink-0">
                    <i class="fas fa-exclamation-triangle text-yellow-500 text-lg"></i>
                </div>
                <div class="ml-3">
                    <h3 class="text-sm font-medium text-yellow-800">Important Notes</h3>
                    <div class="mt-2 text-sm text-yellow-700">
                        <ul class="list-disc pl-5 space-y-1">
                            <li>Submit leave requests at least 3 days in advance</li>
                            <li>Emergency leaves require immediate supervisor notification</li>
                            <li>Provide supporting documents for sick leaves if requested</li>
                            <li>Leave balance: {{ $availableLeaves }} days available out of {{ $totalLeaves }} total</li>
                        </ul>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const startDateInput = document.getElementById('start_date');
    const endDateInput = document.getElementById('end_date');
    const durationPreview = document.getElementById('durationPreview');
    const daysCount = document.getElementById('daysCount');
    const dateRange = document.getElementById('dateRange');
    const leaveBalance = document.getElementById('leaveBalance');
    const reasonTextarea = document.getElementById('reason');
    const charCount = document.getElementById('charCount');
    
    const availableLeaves = {{ $availableLeaves }};
    const totalLeaves = {{ $totalLeaves }};
    
    // Calculate leave duration
    function calculateDuration() {
        const startDate = new Date(startDateInput.value);
        const endDate = new Date(endDateInput.value);
        
        if (startDateInput.value && endDateInput.value && startDate <= endDate) {
            const timeDiff = endDate.getTime() - startDate.getTime();
            const daysDiff = Math.floor(timeDiff / (1000 * 3600 * 24)) + 1;
            
            durationPreview.classList.remove('hidden');
            daysCount.textContent = daysDiff + ' day' + (daysDiff !== 1 ? 's' : '');
            dateRange.textContent = startDate.toLocaleDateString() + ' - ' + endDate.toLocaleDateString();
            
            if (daysDiff > availableLeaves) {
                leaveBalance.innerHTML = '<span class="text-red-600 font-semibold">Insufficient leave balance!</span>';
                leaveBalance.classList.add('text-red-600');
            } else {
                const remaining = availableLeaves - daysDiff;
                leaveBalance.textContent = remaining + ' days remaining after leave';
                leaveBalance.className = 'text-xs ' + (remaining < 5 ? 'text-yellow-600' : 'text-green-600');
            }
        } else {
            durationPreview.classList.add('hidden');
        }
    }
    
    // Calculate character count
    function updateCharCount() {
        charCount.textContent = reasonTextarea.value.length + ' characters';
    }
    
    // Set minimum date to today
    const today = new Date().toISOString().split('T')[0];
    startDateInput.min = today;
    endDateInput.min = today;
    
    // Event listeners
    startDateInput.addEventListener('change', function() {
        endDateInput.min = this.value;
        calculateDuration();
    });
    
    endDateInput.addEventListener('change', calculateDuration);
    reasonTextarea.addEventListener('input', updateCharCount);
    
    // Initialize
    updateCharCount();
});
</script>
@endsection