@extends('layouts.dashboard')

@section('title', $reportData['title'])

@section('content')
<div class="container mx-auto px-4 py-6">
    <!-- Report Header -->
    <div class="mb-8">
        <div class="flex flex-col md:flex-row md:items-center md:justify-between">
            <div>
                <a href="{{ route('reports.index') }}" 
                   class="text-blue-600 hover:text-blue-700 transition-colors duration-200 inline-flex items-center mb-2 group">
                    <i class="fas fa-arrow-left mr-2 group-hover:-translate-x-1 transition-transform duration-200"></i> 
                    Back to Reports
                </a>
                <h1 class="text-2xl font-bold text-gray-900">{{ $reportData['title'] }}</h1>
                <p class="text-gray-600 mt-1 flex flex-wrap items-center gap-2">
                    <span class="flex items-center">
                        <i class="fas fa-calendar-alt mr-1.5"></i> Period: {{ $reportData['period'] }}
                    </span>
                    <span class="hidden md:inline">•</span>
                    <span class="flex items-center">
                        <i class="fas fa-clock mr-1.5"></i> Generated: {{ now()->format('M d, Y h:i A') }}
                    </span>
                </p>
            </div>
            <div class="flex flex-wrap items-center gap-3 mt-4 md:mt-0">
                <button onclick="window.print()" 
                        class="inline-flex items-center px-4 py-2.5 bg-gray-100 hover:bg-gray-200 text-gray-700 font-medium rounded-lg transition duration-200">
                    <i class="fas fa-print mr-2"></i> Print
                </button>
                
                <div class="relative group" x-data="{ open: false }">
                    <button @click="open = !open" 
                            class="inline-flex items-center px-4 py-2.5 bg-blue-600 hover:bg-blue-700 text-white font-medium rounded-lg transition duration-200 shadow-sm hover:shadow-md">
                        <i class="fas fa-download mr-2"></i> Export
                    </button>
                    
                    <div x-show="open" 
                         @click.away="open = false"
                         class="absolute right-0 mt-2 w-48 bg-white rounded-xl shadow-lg border border-gray-200 py-2 z-10"
                         x-transition:enter="transition ease-out duration-200"
                         x-transition:enter-start="opacity-0 scale-95"
                         x-transition:enter-end="opacity-100 scale-100">
                        <a href="{{ route('reports.export', array_merge($request->all(), ['format' => 'excel'])) }}" 
                           class="flex items-center px-4 py-2.5 text-sm text-gray-700 hover:bg-gray-100 transition duration-150">
                            <i class="fas fa-file-excel text-emerald-500 mr-3"></i>
                            Export as Excel
                        </a>
                        <a href="{{ route('reports.export', array_merge($request->all(), ['format' => 'pdf'])) }}" 
                           class="flex items-center px-4 py-2.5 text-sm text-gray-700 hover:bg-gray-100 transition duration-150">
                            <i class="fas fa-file-pdf text-red-500 mr-3"></i>
                            Export as PDF
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Report Summary Stats -->
    <div class="grid grid-cols-2 md:grid-cols-4 gap-4 mb-8">
        @php
            $stats = $reportData['stats'] ?? [];
            $statColors = ['blue', 'emerald', 'amber', 'purple', 'red', 'indigo'];
            $statIcons = ['users', 'check-circle', 'calendar-alt', 'money-bill-wave', 'handshake', 'chart-line'];
        @endphp
        
        @foreach($stats as $key => $value)
            @if(is_numeric($value))
            <div class="bg-white rounded-xl shadow-sm border border-gray-200 p-4 hover:shadow-md transition-shadow duration-200">
                <div class="flex items-center">
                    <div class="p-2.5 rounded-lg bg-{{ $statColors[$loop->index % 6] }}-50 text-{{ $statColors[$loop->index % 6] }}-600">
                        <i class="fas fa-{{ $statIcons[$loop->index % 6] }} text-lg"></i>
                    </div>
                    <div class="ml-3">
                        <p class="text-sm font-medium text-gray-600">{{ ucfirst(str_replace('_', ' ', $key)) }}</p>
                        <p class="text-xl font-bold text-gray-900">{{ number_format($value) }}</p>
                    </div>
                </div>
            </div>
            @endif
        @endforeach
    </div>

    <!-- Report Content -->
    <div class="bg-white rounded-xl shadow-sm border border-gray-200 overflow-hidden mb-8">
        @if($reportData['type'] == 'users')
            <!-- User Report Content -->
            <div class="p-6">
                <div class="flex items-center mb-6">
                    <div class="w-10 h-10 rounded-lg bg-blue-100 text-blue-600 flex items-center justify-center mr-3">
                        <i class="fas fa-users"></i>
                    </div>
                    <h2 class="text-lg font-semibold text-gray-900">User Statistics Report</h2>
                </div>
                
                <!-- Department Distribution -->
                @if(isset($stats['by_department']) && count($stats['by_department']) > 0)
                <div class="mb-8">
                    <h3 class="font-medium text-gray-700 mb-4 flex items-center">
                        <i class="fas fa-building mr-2 text-blue-500"></i>Department Distribution
                    </h3>
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                        @foreach($stats['by_department'] as $dept => $count)
                        <div class="flex items-center justify-between p-4 bg-gradient-to-r from-blue-50 to-white rounded-xl border border-blue-100">
                            <div class="flex items-center">
                                <div class="w-10 h-10 rounded-lg bg-blue-100 text-blue-600 flex items-center justify-center mr-3">
                                    <i class="fas fa-users"></i>
                                </div>
                                <span class="font-medium text-gray-900">{{ $dept ?: 'No Department' }}</span>
                            </div>
                            <span class="px-3 py-1.5 bg-blue-100 text-blue-800 rounded-full text-sm font-medium">
                                {{ $count }} users
                            </span>
                        </div>
                        @endforeach
                    </div>
                </div>
                @endif

                <!-- User List Table -->
                <div class="overflow-x-auto">
                    <table class="min-w-full divide-y divide-gray-200">
                        <thead class="bg-gradient-to-r from-blue-50 to-blue-100">
                            <tr>
                                <th class="px-6 py-3.5 text-left text-xs font-semibold text-blue-700 uppercase tracking-wider">Employee</th>
                                <th class="px-6 py-3.5 text-left text-xs font-semibold text-blue-700 uppercase tracking-wider">Department</th>
                                <th class="px-6 py-3.5 text-left text-xs font-semibold text-blue-700 uppercase tracking-wider">Role</th>
                                <th class="px-6 py-3.5 text-left text-xs font-semibold text-blue-700 uppercase tracking-wider">Status</th>
                                <th class="px-6 py-3.5 text-left text-xs font-semibold text-blue-700 uppercase tracking-wider">Join Date</th>
                            </tr>
                        </thead>
                        <tbody class="divide-y divide-gray-200">
                            @foreach($reportData['data'] as $user)
                            <tr class="hover:bg-gray-50 transition-colors duration-150">
                                <td class="px-6 py-4">
                                    <div class="flex items-center">
                                        <div class="flex-shrink-0">
                                            <div class="h-9 w-9 rounded-full bg-gradient-to-br from-blue-500 to-indigo-600 flex items-center justify-center shadow-sm">
                                                <i class="fas fa-user text-white text-xs"></i>
                                            </div>
                                        </div>
                                        <div class="ml-3">
                                            <div class="text-sm font-medium text-gray-900">{{ $user->name }}</div>
                                            <div class="text-xs text-gray-500">{{ $user->email }}</div>
                                        </div>
                                    </div>
                                </td>
                                <td class="px-6 py-4 text-sm text-gray-900">{{ $user->department->name ?? 'N/A' }}</td>
                                <td class="px-6 py-4">
                                    <span class="px-2.5 py-1 text-xs rounded-full bg-gray-100 text-gray-800 font-medium">
                                        {{ ucfirst($user->role) }}
                                    </span>
                                </td>
                                <td class="px-6 py-4">
                                    @if($user->status == 'active')
                                    <span class="px-2.5 py-1 text-xs rounded-full bg-emerald-100 text-emerald-800 font-medium">
                                        Active
                                    </span>
                                    @else
                                    <span class="px-2.5 py-1 text-xs rounded-full bg-red-100 text-red-800 font-medium">
                                        Inactive
                                    </span>
                                    @endif
                                </td>
                                <td class="px-6 py-4 text-sm text-gray-500">{{ $user->created_at->format('M d, Y') }}</td>
                            </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>
            </div>

        @elseif($reportData['type'] == 'visits')
            <!-- Visit Report Content -->
            <div class="p-6">
                <div class="flex items-center mb-6">
                    <div class="w-10 h-10 rounded-lg bg-emerald-100 text-emerald-600 flex items-center justify-center mr-3">
                        <i class="fas fa-map-marker-alt"></i>
                    </div>
                    <h2 class="text-lg font-semibold text-gray-900">Visit Analytics Report</h2>
                </div>
                
                <!-- Status Distribution -->
                <div class="grid grid-cols-2 md:grid-cols-5 gap-4 mb-8">
                    @foreach(['scheduled', 'in_progress', 'completed', 'cancelled'] as $status)
                    <div class="bg-gradient-to-br from-gray-50 to-white rounded-xl p-4 text-center border border-gray-200">
                        <div class="text-2xl font-bold text-gray-900">
                            {{ $stats[$status . '_visits'] ?? 0 }}
                        </div>
                        <div class="text-sm text-gray-600 capitalize mt-1">{{ str_replace('_', ' ', $status) }}</div>
                    </div>
                    @endforeach
                    <div class="bg-gradient-to-br from-blue-50 to-blue-100 rounded-xl p-4 text-center border border-blue-200">
                        <div class="text-2xl font-bold text-blue-600">
                            {{ $stats['avg_completion_time'] ?? 0 }} min
                        </div>
                        <div class="text-sm text-blue-600 mt-1">Avg. Duration</div>
                    </div>
                </div>

                <!-- Visit List Table -->
                <div class="overflow-x-auto">
                    <table class="min-w-full divide-y divide-gray-200">
                        <thead class="bg-gradient-to-r from-emerald-50 to-emerald-100">
                            <tr>
                                <th class="px-6 py-3.5 text-left text-xs font-semibold text-emerald-700 uppercase tracking-wider">Employee</th>
                                <th class="px-6 py-3.5 text-left text-xs font-semibold text-emerald-700 uppercase tracking-wider">Client</th>
                                <th class="px-6 py-3.5 text-left text-xs font-semibold text-emerald-700 uppercase tracking-wider">Location</th>
                                <th class="px-6 py-3.5 text-left text-xs font-semibold text-emerald-700 uppercase tracking-wider">Date & Time</th>
                                <th class="px-6 py-3.5 text-left text-xs font-semibold text-emerald-700 uppercase tracking-wider">Status</th>
                                <th class="px-6 py-3.5 text-left text-xs font-semibold text-emerald-700 uppercase tracking-wider">Duration</th>
                            </tr>
                        </thead>
                        <tbody class="divide-y divide-gray-200">
                            @foreach($reportData['data'] as $visit)
                            <tr class="hover:bg-gray-50 transition-colors duration-150">
                                <td class="px-6 py-4">
                                    <div class="text-sm font-medium text-gray-900">{{ $visit->employee->name }}</div>
                                    <div class="text-xs text-gray-500">{{ $visit->employee->employee_id }}</div>
                                </td>
                                <td class="px-6 py-4">
                                    <div class="text-sm font-medium text-gray-900">{{ $visit->client->name }}</div>
                                    <div class="text-xs text-gray-500">{{ $visit->client->company_name }}</div>
                                </td>
                                <td class="px-6 py-4">
                                    <div class="text-sm text-gray-900">{{ $visit->governorate->name ?? '' }}</div>
                                    <div class="text-xs text-gray-500">{{ $visit->city->name ?? '' }}</div>
                                </td>
                                <td class="px-6 py-4">
                                    <div class="text-sm text-gray-900">{{ $visit->scheduled_date->format('M d, Y') }}</div>
                                    <div class="text-xs text-gray-500">{{ $visit->scheduled_time }}</div>
                                </td>
                                <td class="px-6 py-4">
                                    @php
                                        $statusColors = [
                                            'scheduled' => 'bg-amber-100 text-amber-800',
                                            'in_progress' => 'bg-blue-100 text-blue-800',
                                            'completed' => 'bg-emerald-100 text-emerald-800',
                                            'cancelled' => 'bg-red-100 text-red-800'
                                        ];
                                    @endphp
                                    <span class="px-2.5 py-1 text-xs rounded-full font-medium {{ $statusColors[$visit->status] }}">
                                        {{ ucfirst(str_replace('_', ' ', $visit->status)) }}
                                    </span>
                                </td>
                                <td class="px-6 py-4">
                                    @if($visit->status == 'completed' && $visit->started_at && $visit->completed_at)
                                    <div class="text-sm font-medium text-gray-900">
                                        {{ $visit->started_at->diffInMinutes($visit->completed_at) }} min
                                    </div>
                                    @else
                                    <div class="text-sm text-gray-400">-</div>
                                    @endif
                                </td>
                            </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>
            </div>

        @elseif($reportData['type'] == 'leaves')
            <!-- Leave Report Content -->
            <div class="p-6">
                <div class="flex items-center mb-6">
                    <div class="w-10 h-10 rounded-lg bg-amber-100 text-amber-600 flex items-center justify-center mr-3">
                        <i class="fas fa-calendar-alt"></i>
                    </div>
                    <h2 class="text-lg font-semibold text-gray-900">Leave Management Report</h2>
                </div>
                
                <!-- Summary Stats -->
                <div class="grid grid-cols-2 md:grid-cols-4 gap-4 mb-8">
                    <div class="bg-gradient-to-br from-emerald-50 to-emerald-100 rounded-xl p-4 text-center border border-emerald-200">
                        <div class="text-2xl font-bold text-emerald-600">{{ $stats['approved_leaves'] ?? 0 }}</div>
                        <div class="text-sm text-emerald-600 font-medium mt-1">Approved</div>
                    </div>
                    <div class="bg-gradient-to-br from-amber-50 to-amber-100 rounded-xl p-4 text-center border border-amber-200">
                        <div class="text-2xl font-bold text-amber-600">{{ $stats['pending_leaves'] ?? 0 }}</div>
                        <div class="text-sm text-amber-600 font-medium mt-1">Pending</div>
                    </div>
                    <div class="bg-gradient-to-br from-red-50 to-red-100 rounded-xl p-4 text-center border border-red-200">
                        <div class="text-2xl font-bold text-red-600">{{ $stats['rejected_leaves'] ?? 0 }}</div>
                        <div class="text-sm text-red-600 font-medium mt-1">Rejected</div>
                    </div>
                    <div class="bg-gradient-to-br from-blue-50 to-blue-100 rounded-xl p-4 text-center border border-blue-200">
                        <div class="text-2xl font-bold text-blue-600">{{ $stats['total_days'] ?? 0 }}</div>
                        <div class="text-sm text-blue-600 font-medium mt-1">Total Days</div>
                    </div>
                </div>

                <!-- Leave List Table -->
                <div class="overflow-x-auto">
                    <table class="min-w-full divide-y divide-gray-200">
                        <thead class="bg-gradient-to-r from-amber-50 to-amber-100">
                            <tr>
                                <th class="px-6 py-3.5 text-left text-xs font-semibold text-amber-700 uppercase tracking-wider">Employee</th>
                                <th class="px-6 py-3.5 text-left text-xs font-semibold text-amber-700 uppercase tracking-wider">Leave Type</th>
                                <th class="px-6 py-3.5 text-left text-xs font-semibold text-amber-700 uppercase tracking-wider">Period</th>
                                <th class="px-6 py-3.5 text-left text-xs font-semibold text-amber-700 uppercase tracking-wider">Days</th>
                                <th class="px-6 py-3.5 text-left text-xs font-semibold text-amber-700 uppercase tracking-wider">Status</th>
                                <th class="px-6 py-3.5 text-left text-xs font-semibold text-amber-700 uppercase tracking-wider">Reason</th>
                            </tr>
                        </thead>
                        <tbody class="divide-y divide-gray-200">
                            @foreach($reportData['data'] as $leave)
                            <tr class="hover:bg-gray-50 transition-colors duration-150">
                                <td class="px-6 py-4">
                                    <div class="text-sm font-medium text-gray-900">{{ $leave->employee->name }}</div>
                                    <div class="text-xs text-gray-500">{{ $leave->employee->department->name ?? 'N/A' }}</div>
                                </td>
                                <td class="px-6 py-4">
                                    <span class="px-2.5 py-1 text-xs rounded-full bg-gray-100 text-gray-800 font-medium">
                                        {{ ucfirst($leave->type) }}
                                    </span>
                                </td>
                                <td class="px-6 py-4">
                                    <div class="text-sm text-gray-900">{{ $leave->start_date->format('M d') }} - {{ $leave->end_date->format('M d, Y') }}</div>
                                </td>
                                <td class="px-6 py-4">
                                    <div class="text-sm font-medium text-gray-900">{{ $leave->total_days }}</div>
                                </td>
                                <td class="px-6 py-4">
                                    @php
                                        $statusColors = [
                                            'pending' => 'bg-amber-100 text-amber-800',
                                            'approved' => 'bg-emerald-100 text-emerald-800',
                                            'rejected' => 'bg-red-100 text-red-800'
                                        ];
                                    @endphp
                                    <span class="px-2.5 py-1 text-xs rounded-full font-medium {{ $statusColors[$leave->status] }}">
                                        {{ ucfirst($leave->status) }}
                                    </span>
                                </td>
                                <td class="px-6 py-4">
                                    <div class="text-sm text-gray-900 truncate max-w-xs">{{ $leave->reason }}</div>
                                </td>
                            </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>
            </div>

        @elseif($reportData['type'] == 'payroll')
            <!-- Payroll Report Content -->
            <div class="p-6">
                <div class="flex items-center mb-6">
                    <div class="w-10 h-10 rounded-lg bg-purple-100 text-purple-600 flex items-center justify-center mr-3">
                        <i class="fas fa-money-bill-wave"></i>
                    </div>
                    <h2 class="text-lg font-semibold text-gray-900">Payroll Management Report</h2>
                </div>
                
                <!-- Financial Summary -->
                <div class="grid grid-cols-1 md:grid-cols-3 gap-4 mb-8">
                    <div class="bg-gradient-to-br from-blue-50 to-blue-100 rounded-xl p-5 border border-blue-200">
                        <div class="text-sm text-gray-600 mb-2">Total Salary Paid</div>
                        <div class="text-2xl font-bold text-blue-600">
                            {{ number_format($stats['total_salary_paid'] ?? 0, 2) }} EGP
                        </div>
                    </div>
                    <div class="bg-gradient-to-br from-emerald-50 to-emerald-100 rounded-xl p-5 border border-emerald-200">
                        <div class="text-sm text-gray-600 mb-2">Average Salary</div>
                        <div class="text-2xl font-bold text-emerald-600">
                            {{ number_format($stats['avg_salary'] ?? 0, 2) }} EGP
                        </div>
                    </div>
                    <div class="bg-gradient-to-br from-purple-50 to-purple-100 rounded-xl p-5 border border-purple-200">
                        <div class="text-sm text-gray-600 mb-2">Payrolls Generated</div>
                        <div class="text-2xl font-bold text-purple-600">{{ $stats['total_payrolls'] ?? 0 }}</div>
                    </div>
                </div>

                <!-- Payroll List Table -->
                <div class="overflow-x-auto">
                    <table class="min-w-full divide-y divide-gray-200">
                        <thead class="bg-gradient-to-r from-purple-50 to-purple-100">
                            <tr>
                                <th class="px-6 py-3.5 text-left text-xs font-semibold text-purple-700 uppercase tracking-wider">Employee</th>
                                <th class="px-6 py-3.5 text-left text-xs font-semibold text-purple-700 uppercase tracking-wider">Pay Period</th>
                                <th class="px-6 py-3.5 text-left text-xs font-semibold text-purple-700 uppercase tracking-wider">Basic Salary</th>
                                <th class="px-6 py-3.5 text-left text-xs font-semibold text-purple-700 uppercase tracking-wider">Allowances</th>
                                <th class="px-6 py-3.5 text-left text-xs font-semibold text-purple-700 uppercase tracking-wider">Deductions</th>
                                <th class="px-6 py-3.5 text-left text-xs font-semibold text-purple-700 uppercase tracking-wider">Net Salary</th>
                                <th class="px-6 py-3.5 text-left text-xs font-semibold text-purple-700 uppercase tracking-wider">Status</th>
                                <th class="px-6 py-3.5 text-left text-xs font-semibold text-purple-700 uppercase tracking-wider">Payment Date</th>
                            </tr>
                        </thead>
                        <tbody class="divide-y divide-gray-200">
                            @foreach($reportData['data'] as $payroll)
                            <tr class="hover:bg-gray-50 transition-colors duration-150">
                                <td class="px-6 py-4">
                                    <div class="text-sm font-medium text-gray-900">{{ $payroll->employee->name }}</div>
                                </td>
                                <td class="px-6 py-4">
                                    <div class="text-sm text-gray-900">
                                        {{ $payroll->pay_period_start->format('M d') }} - {{ $payroll->pay_period_end->format('M d, Y') }}
                                    </div>
                                </td>
                                <td class="px-6 py-4 text-sm text-gray-900">
                                    {{ number_format($payroll->basic_salary, 2) }}
                                </td>
                                <td class="px-6 py-4 text-sm font-medium text-emerald-600">
                                    {{ number_format($payroll->allowances, 2) }}
                                </td>
                                <td class="px-6 py-4 text-sm font-medium text-red-600">
                                    {{ number_format($payroll->deductions, 2) }}
                                </td>
                                <td class="px-6 py-4">
                                    <div class="text-sm font-bold text-blue-600">
                                        {{ number_format($payroll->net_salary, 2) }}
                                    </div>
                                </td>
                                <td class="px-6 py-4">
                                    @php
                                        $statusColors = [
                                            'draft' => 'bg-amber-100 text-amber-800',
                                            'processed' => 'bg-blue-100 text-blue-800',
                                            'paid' => 'bg-emerald-100 text-emerald-800'
                                        ];
                                    @endphp
                                    <span class="px-2.5 py-1 text-xs rounded-full font-medium {{ $statusColors[$payroll->status] }}">
                                        {{ ucfirst($payroll->status) }}
                                    </span>
                                </td>
                                <td class="px-6 py-4">
                                    @if($payroll->payment_date)
                                    <div class="text-sm text-gray-900">{{ $payroll->payment_date->format('M d, Y') }}</div>
                                    @else
                                    <div class="text-sm text-gray-400">Not scheduled</div>
                                    @endif
                                </td>
                            </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>
            </div>

        @elseif($reportData['type'] == 'clients')
            <!-- Client Report Content -->
            <div class="p-6">
                <div class="flex items-center mb-6">
                    <div class="w-10 h-10 rounded-lg bg-red-100 text-red-600 flex items-center justify-center mr-3">
                        <i class="fas fa-handshake"></i>
                    </div>
                    <h2 class="text-lg font-semibold text-gray-900">Client Management Report</h2>
                </div>
                
                <!-- Client Distribution -->
                @if(isset($stats['by_governorate']) && count($stats['by_governorate']) > 0)
                <div class="mb-8">
                    <h3 class="font-medium text-gray-700 mb-4 flex items-center">
                        <i class="fas fa-map mr-2 text-red-500"></i>Client Distribution by Governorate
                    </h3>
                    <div class="space-y-3">
                        @foreach($stats['by_governorate'] as $gov => $count)
                        <div class="flex items-center">
                            <div class="w-32 text-sm font-medium text-gray-600">{{ $gov ?: 'Unknown' }}</div>
                            <div class="flex-1 ml-4">
                                <div class="h-4 bg-red-100 rounded-full overflow-hidden">
                                    <div class="h-full bg-gradient-to-r from-red-400 to-red-500" 
                                         style="width: {{ ($count / max($stats['by_governorate']->toArray())) * 100 }}%"></div>
                                </div>
                            </div>
                            <div class="w-16 text-right text-sm font-bold text-red-600">{{ $count }}</div>
                        </div>
                        @endforeach
                    </div>
                </div>
                @endif

                <!-- Client List Table -->
                <div class="overflow-x-auto">
                    <table class="min-w-full divide-y divide-gray-200">
                        <thead class="bg-gradient-to-r from-red-50 to-red-100">
                            <tr>
                                <th class="px-6 py-3.5 text-left text-xs font-semibold text-red-700 uppercase tracking-wider">Client</th>
                                <th class="px-6 py-3.5 text-left text-xs font-semibold text-red-700 uppercase tracking-wider">Company</th>
                                <th class="px-6 py-3.5 text-left text-xs font-semibold text-red-700 uppercase tracking-wider">Location</th>
                                <th class="px-6 py-3.5 text-left text-xs font-semibold text-red-700 uppercase tracking-wider">Contact</th>
                                <th class="px-6 py-3.5 text-left text-xs font-semibold text-red-700 uppercase tracking-wider">Type</th>
                                <th class="px-6 py-3.5 text-left text-xs font-semibold text-red-700 uppercase tracking-wider">Status</th>
                                <th class="px-6 py-3.5 text-left text-xs font-semibold text-red-700 uppercase tracking-wider">Total Visits</th>
                            </tr>
                        </thead>
                        <tbody class="divide-y divide-gray-200">
                            @foreach($reportData['data'] as $client)
                            <tr class="hover:bg-gray-50 transition-colors duration-150">
                                <td class="px-6 py-4">
                                    <div class="text-sm font-medium text-gray-900">{{ $client->name }}</div>
                                </td>
                                <td class="px-6 py-4">
                                    <div class="text-sm text-gray-900">{{ $client->company_name }}</div>
                                </td>
                                <td class="px-6 py-4">
                                    <div class="text-sm text-gray-900">{{ $client->governorate->name ?? '' }}</div>
                                    <div class="text-xs text-gray-500">{{ $client->city->name ?? '' }}, {{ $client->area }}</div>
                                </td>
                                <td class="px-6 py-4">
                                    <div class="text-sm text-gray-900">{{ $client->phone }}</div>
                                    <div class="text-xs text-gray-500">{{ $client->email }}</div>
                                </td>
                                <td class="px-6 py-4">
                                    <span class="px-2.5 py-1 text-xs rounded-full bg-gray-100 text-gray-800 font-medium">
                                        {{ ucfirst($client->type) }}
                                    </span>
                                </td>
                                <td class="px-6 py-4">
                                    @if($client->status == 'active')
                                    <span class="px-2.5 py-1 text-xs rounded-full bg-emerald-100 text-emerald-800 font-medium">
                                        Active
                                    </span>
                                    @else
                                    <span class="px-2.5 py-1 text-xs rounded-full bg-red-100 text-red-800 font-medium">
                                        Inactive
                                    </span>
                                    @endif
                                </td>
                                <td class="px-6 py-4">
                                    <div class="text-sm font-medium text-gray-900">{{ $client->visits()->count() }}</div>
                                </td>
                            </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>
            </div>

        @elseif($reportData['type'] == 'performance')
            <!-- Performance Report Content -->
            <div class="p-6">
                <div class="flex items-center mb-6">
                    <div class="w-10 h-10 rounded-lg bg-indigo-100 text-indigo-600 flex items-center justify-center mr-3">
                        <i class="fas fa-chart-line"></i>
                    </div>
                    <h2 class="text-lg font-semibold text-gray-900">Employee Performance Report</h2>
                </div>
                
                <!-- Top Performers -->
                @if(isset($stats['top_performers']) && count($stats['top_performers']) > 0)
                <div class="mb-8">
                    <h3 class="font-medium text-gray-700 mb-4 flex items-center">
                        <i class="fas fa-trophy mr-2 text-amber-500"></i>Top Performers
                    </h3>
                    <div class="space-y-3">
                        @foreach($stats['top_performers'] as $performer)
                        <div class="flex items-center justify-between p-4 bg-gradient-to-r from-emerald-50 to-white rounded-xl border border-emerald-100">
                            <div class="flex items-center">
                                <div class="flex-shrink-0">
                                    <div class="h-10 w-10 rounded-full bg-gradient-to-br from-emerald-400 to-emerald-500 flex items-center justify-center shadow-sm">
                                        <i class="fas fa-user text-white"></i>
                                    </div>
                                </div>
                                <div class="ml-3">
                                    <div class="text-sm font-medium text-gray-900">{{ $performer['employee']['name'] }}</div>
                                    <div class="text-xs text-gray-500">{{ $performer['department'] }}</div>
                                </div>
                            </div>
                            <div class="text-right">
                                <div class="text-lg font-bold text-emerald-600">{{ number_format($performer['completion_rate'], 1) }}%</div>
                                <div class="text-xs text-gray-500">Completion Rate</div>
                            </div>
                        </div>
                        @endforeach
                    </div>
                </div>
                @endif

                <!-- Performance Table -->
                <div class="overflow-x-auto">
                    <table class="min-w-full divide-y divide-gray-200">
                        <thead class="bg-gradient-to-r from-indigo-50 to-indigo-100">
                            <tr>
                                <th class="px-6 py-3.5 text-left text-xs font-semibold text-indigo-700 uppercase tracking-wider">Employee</th>
                                <th class="px-6 py-3.5 text-left text-xs font-semibold text-indigo-700 uppercase tracking-wider">Department</th>
                                <th class="px-6 py-3.5 text-left text-xs font-semibold text-indigo-700 uppercase tracking-wider">Total Visits</th>
                                <th class="px-6 py-3.5 text-left text-xs font-semibold text-indigo-700 uppercase tracking-wider">Completed</th>
                                <th class="px-6 py-3.5 text-left text-xs font-semibold text-indigo-700 uppercase tracking-wider">Completion Rate</th>
                                <th class="px-6 py-3.5 text-left text-xs font-semibold text-indigo-700 uppercase tracking-wider">Leave Days</th>
                                <th class="px-6 py-3.5 text-left text-xs font-semibold text-indigo-700 uppercase tracking-wider">Avg. Duration</th>
                            </tr>
                        </thead>
                        <tbody class="divide-y divide-gray-200">
                            @foreach($reportData['data'] as $performance)
                            <tr class="hover:bg-gray-50 transition-colors duration-150">
                                <td class="px-6 py-4">
                                    <div class="text-sm font-medium text-gray-900">{{ $performance['employee']['name'] }}</div>
                                </td>
                                <td class="px-6 py-4">
                                    <div class="text-sm text-gray-900">{{ $performance['department'] }}</div>
                                </td>
                                <td class="px-6 py-4">
                                    <div class="text-sm font-medium text-gray-900">{{ $performance['total_visits'] }}</div>
                                </td>
                                <td class="px-6 py-4">
                                    <div class="text-sm font-medium text-emerald-600">{{ $performance['completed_visits'] }}</div>
                                </td>
                                <td class="px-6 py-4">
                                    @php
                                        $rate = $performance['completion_rate'];
                                        if($rate >= 90) {
                                            $color = 'bg-emerald-100 text-emerald-800';
                                        } elseif($rate >= 70) {
                                            $color = 'bg-amber-100 text-amber-800';
                                        } else {
                                            $color = 'bg-red-100 text-red-800';
                                        }
                                    @endphp
                                    <span class="px-2.5 py-1 text-xs rounded-full font-medium {{ $color }}">
                                        {{ number_format($rate, 1) }}%
                                    </span>
                                </td>
                                <td class="px-6 py-4">
                                    <div class="text-sm text-gray-900">{{ $performance['total_leave_days'] }}</div>
                                </td>
                                <td class="px-6 py-4">
                                    <div class="text-sm text-gray-900">{{ $performance['avg_visit_duration'] }} min</div>
                                </td>
                            </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>
            </div>

        @elseif($reportData['type'] == 'summary')
            <!-- Summary Report Content -->
            <div class="p-6">
                <div class="flex items-center mb-6">
                    <div class="w-10 h-10 rounded-lg bg-gradient-to-br from-blue-100 to-indigo-200 text-blue-600 flex items-center justify-center mr-3">
                        <i class="fas fa-chart-bar"></i>
                    </div>
                    <h2 class="text-lg font-semibold text-gray-900">System Summary Report</h2>
                </div>
                
                <!-- Overall Stats -->
                <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 mb-8">
                    @foreach($reportData['summary'] as $module => $moduleStats)
                    <div class="bg-gradient-to-br from-gray-50 to-white rounded-xl p-5 border border-gray-200">
                        <div class="flex items-center mb-3">
                            @php
                                $moduleIcons = [
                                    'users' => 'fa-users text-blue-500',
                                    'visits' => 'fa-map-marker-alt text-emerald-500',
                                    'leaves' => 'fa-calendar-alt text-amber-500',
                                    'payroll' => 'fa-money-bill-wave text-purple-500',
                                    'clients' => 'fa-handshake text-red-500'
                                ];
                            @endphp
                            <i class="fas {{ $moduleIcons[$module] ?? 'fa-folder text-gray-500' }} text-lg mr-2"></i>
                            <div class="text-sm font-semibold text-gray-700 capitalize">{{ $module }}</div>
                        </div>
                        @foreach($moduleStats as $key => $value)
                            @if(is_numeric($value))
                            <div class="flex justify-between text-sm py-1.5 border-t border-gray-100">
                                <span class="text-gray-600">{{ ucfirst(str_replace('_', ' ', $key)) }}</span>
                                <span class="font-medium text-gray-900">{{ number_format($value) }}</span>
                            </div>
                            @endif
                        @endforeach
                    </div>
                    @endforeach
                </div>

                <!-- Monthly Trends -->
                <div class="mb-8">
                    <h3 class="font-medium text-gray-700 mb-4 flex items-center">
                        <i class="fas fa-chart-line mr-2 text-blue-500"></i> Monthly Trends (Last 6 Months)
                    </h3>
                    <div class="bg-white border border-gray-200 rounded-xl p-5">
                        <div class="overflow-x-auto">
                            <table class="min-w-full">
                                <thead>
                                    <tr class="border-b border-gray-200">
                                        <th class="px-4 py-3 text-left text-sm font-medium text-gray-700">Month</th>
                                        <th class="px-4 py-3 text-left text-sm font-medium text-blue-600">Visits</th>
                                        <th class="px-4 py-3 text-left text-sm font-medium text-amber-600">Leaves</th>
                                        <th class="px-4 py-3 text-left text-sm font-medium text-purple-600">Payrolls</th>
                                        <th class="px-4 py-3 text-left text-sm font-medium text-emerald-600">New Users</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    @foreach($reportData['monthly_trends'] as $month => $trend)
                                    <tr class="border-t border-gray-100 hover:bg-gray-50 transition-colors duration-150">
                                        <td class="px-4 py-3 text-sm font-medium text-gray-900">{{ $month }}</td>
                                        <td class="px-4 py-3 text-sm font-bold text-blue-600">{{ $trend['visits'] }}</td>
                                        <td class="px-4 py-3 text-sm font-bold text-amber-600">{{ $trend['leaves'] }}</td>
                                        <td class="px-4 py-3 text-sm font-bold text-purple-600">{{ $trend['payroll'] }}</td>
                                        <td class="px-4 py-3 text-sm font-bold text-emerald-600">{{ $trend['new_users'] ?? 0 }}</td>
                                    </tr>
                                    @endforeach
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>

                <!-- System Health -->
                <div>
                    <h3 class="font-medium text-gray-700 mb-4 flex items-center">
                        <i class="fas fa-heartbeat mr-2 text-red-500"></i> System Health Status
                    </h3>
                    <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
                        <div class="bg-gradient-to-br from-emerald-50 to-emerald-100 rounded-xl p-5 border border-emerald-200">
                            <div class="flex items-center">
                                <i class="fas fa-check-circle text-emerald-500 text-2xl mr-4"></i>
                                <div>
                                    <div class="font-semibold text-emerald-800">Operational</div>
                                    <div class="text-sm text-emerald-600 mt-1">All systems are running normally</div>
                                </div>
                            </div>
                        </div>
                        <div class="bg-gradient-to-br from-blue-50 to-blue-100 rounded-xl p-5 border border-blue-200">
                            <div class="flex items-center">
                                <i class="fas fa-database text-blue-500 text-2xl mr-4"></i>
                                <div>
                                    <div class="font-semibold text-blue-800">Data Integrity</div>
                                    <div class="text-sm text-blue-600 mt-1">All data is properly synchronized</div>
                                </div>
                            </div>
                        </div>
                        <div class="bg-gradient-to-br from-purple-50 to-purple-100 rounded-xl p-5 border border-purple-200">
                            <div class="flex items-center">
                                <i class="fas fa-shield-alt text-purple-500 text-2xl mr-4"></i>
                                <div>
                                    <div class="font-semibold text-purple-800">Security Status</div>
                                    <div class="text-sm text-purple-600 mt-1">Security protocols are active</div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        @endif
    </div>

    <!-- Export Notice -->
    @if($request->format == 'excel')
    <div class="bg-gradient-to-br from-emerald-50 to-emerald-100 border border-emerald-200 rounded-xl p-5 mb-8">
        <div class="flex items-center">
            <i class="fas fa-file-excel text-emerald-500 text-2xl mr-4"></i>
            <div>
                <div class="font-semibold text-emerald-800">Excel Export Ready</div>
                <div class="text-sm text-emerald-600 mt-1">Your report has been exported to Excel format. The download should start automatically.</div>
            </div>
        </div>
    </div>
    @endif

    <!-- Report Footer -->
    <div class="text-center text-sm text-gray-500 py-6 border-t border-gray-200">
        <p class="font-medium text-gray-700">Report generated by {{ auth()->user()->name }} on {{ now()->format('F d, Y \a\t h:i A') }}</p>
        <p class="mt-1 text-gray-500">Innovaxcess Attendance System - Confidential Report</p>
        <p class="mt-1 text-gray-400 text-xs">Report ID: {{ Str::uuid()->toString() }}</p>
    </div>
</div>

<!-- Print Styles -->
<style>
@media print {
    nav, .btn-primary, .btn-secondary, [x-data], [onclick], .text-center:last-child,
    .flex:last-child, .hidden, [x-show] {
        display: none !important;
    }
    
    body {
        background: white !important;
        font-size: 11px !important;
    }
    
    .container {
        max-width: 100% !important;
        padding: 0 !important;
        margin: 0 !important;
    }
    
    .rounded-xl, .shadow-sm, .border {
        border: 1px solid #ddd !important;
        box-shadow: none !important;
        border-radius: 0 !important;
    }
    
    .bg-white {
        background: white !important;
    }
    
    .text-gray-900, .text-gray-800 {
        color: #111827 !important;
    }
    
    .grid {
        display: block !important;
    }
    
    .space-y-6 > * {
        margin-bottom: 20px !important;
        page-break-inside: avoid;
    }
    
    table {
        font-size: 9px !important;
        border-collapse: collapse !important;
        width: 100% !important;
    }
    
    th, td {
        padding: 6px 8px !important;
        border: 1px solid #ddd !important;
    }
    
    .px-6 {
        padding-left: 8px !important;
        padding-right: 8px !important;
    }
    
    .py-4 {
        padding-top: 6px !important;
        padding-bottom: 6px !important;
    }
    
    h1, h2, h3 {
        page-break-after: avoid !important;
    }
    
    table {
        page-break-inside: auto !important;
    }
    
    tr {
        page-break-inside: avoid !important;
        page-break-after: auto !important;
    }
    
    /* Remove gradients for printing */
    .bg-gradient-to-r {
        background: #f9fafb !important;
    }
}
</style>

<script>
// Auto-download Excel if requested
document.addEventListener('DOMContentLoaded', function() {
    @if($request->format == 'excel')
    setTimeout(() => {
        // Show download success message
        const notice = document.createElement('div');
        notice.className = 'fixed bottom-4 right-4 bg-emerald-500 text-white p-4 rounded-xl shadow-lg z-50';
        notice.innerHTML = `
            <div class="flex items-center">
                <i class="fas fa-check-circle text-xl mr-3"></i>
                <div>
                    <div class="font-semibold">Excel report downloaded!</div>
                    <div class="text-sm opacity-90">Check your downloads folder</div>
                </div>
            </div>
        `;
        document.body.appendChild(notice);
        
        setTimeout(() => {
            notice.style.opacity = '0';
            setTimeout(() => {
                notice.remove();
            }, 300);
        }, 5000);
    }, 1000);
    @endif
    
    // Print functionality
    const printBtn = document.querySelector('[onclick="window.print()"]');
    if (printBtn) {
        printBtn.addEventListener('click', function() {
            window.print();
        });
    }
    
    // Smooth scroll for report sections
    document.querySelectorAll('a[href^="#"]').forEach(anchor => {
        anchor.addEventListener('click', function (e) {
            e.preventDefault();
            const target = document.querySelector(this.getAttribute('href'));
            if (target) {
                target.scrollIntoView({ behavior: 'smooth', block: 'start' });
            }
        });
    });
});
</script>
@endsection