@extends('layouts.app')

@section('content')
<div class="max-w-4xl mx-auto py-8 px-4">
    <div class="bg-white rounded-xl shadow-lg p-6">
        <h1 class="text-2xl font-bold text-gray-800 mb-6">Clear Browser Cache</h1>
        
        <div class="mb-6">
            <p class="text-gray-600 mb-4">If you're experiencing issues with stale content, you can clear the browser cache:</p>
            
            <div class="bg-yellow-50 border-l-4 border-yellow-400 p-4 mb-6">
                <div class="flex">
                    <div class="flex-shrink-0">
                        <i class="fas fa-exclamation-triangle text-yellow-400"></i>
                    </div>
                    <div class="ml-3">
                        <p class="text-sm text-yellow-700">
                            This will clear the service worker cache and force a fresh load of all assets.
                        </p>
                    </div>
                </div>
            </div>
        </div>
        
        <div class="space-y-4">
            <button onclick="clearServiceWorkerCache()" 
                    class="btn-primary flex items-center space-x-2">
                <i class="fas fa-broom"></i>
                <span>Clear Service Worker Cache</span>
            </button>
            
            <button onclick="forceReload()" 
                    class="btn-secondary flex items-center space-x-2">
                <i class="fas fa-sync-alt"></i>
                <span>Force Hard Reload</span>
            </button>
            
            <button onclick="checkCacheStatus()" 
                    class="bg-gray-100 hover:bg-gray-200 text-gray-800 font-medium py-2 px-4 rounded-lg transition duration-200 flex items-center space-x-2">
                <i class="fas fa-info-circle"></i>
                <span>Check Cache Status</span>
            </button>
        </div>
        
        <div id="cache-status" class="mt-6 p-4 bg-gray-50 rounded-lg hidden">
            <h3 class="font-semibold text-gray-700 mb-2">Cache Status</h3>
            <pre id="cache-info" class="text-sm bg-gray-800 text-gray-100 p-3 rounded overflow-auto"></pre>
        </div>
    </div>
</div>

<script>
// Clear Service Worker Cache
function clearServiceWorkerCache() {
    if ('serviceWorker' in navigator) {
        navigator.serviceWorker.getRegistrations().then(function(registrations) {
            for(let registration of registrations) {
                registration.unregister();
            }
            return caches.keys();
        }).then(function(cacheNames) {
            return Promise.all(
                cacheNames.map(function(cacheName) {
                    return caches.delete(cacheName);
                })
            );
        }).then(function() {
            showToast('Service Worker cache cleared successfully!', 'success');
            // Reload page to get fresh assets
            setTimeout(() => {
                window.location.reload(true);
            }, 1500);
        }).catch(function(error) {
            showToast('Error clearing cache: ' + error.message, 'error');
        });
    } else {
        showToast('Service Worker not supported', 'warning');
    }
}

// Force Hard Reload
function forceReload() {
    // Clear localStorage version
    localStorage.removeItem('app_version');
    
    // Clear sessionStorage
    sessionStorage.clear();
    
    // Force reload bypassing cache
    window.location.reload(true);
}

// Check Cache Status
function checkCacheStatus() {
    if ('serviceWorker' in navigator && navigator.serviceWorker.controller) {
        navigator.serviceWorker.controller.postMessage({ type: 'GET_CACHE_INFO' });
        
        // Listen for response
        navigator.serviceWorker.addEventListener('message', function(event) {
            if (event.data.type === 'CACHE_INFO') {
                document.getElementById('cache-info').textContent = 
                    JSON.stringify(event.data.caches, null, 2);
                document.getElementById('cache-status').classList.remove('hidden');
                showToast('Cache info retrieved', 'success');
            }
        });
    } else {
        showToast('No active Service Worker', 'warning');
    }
    
    // Show localStorage version
    const version = localStorage.getItem('app_version') || 'Not set';
    console.log('App Version:', version);
}

// Listen for cache cleared messages
if ('serviceWorker' in navigator) {
    navigator.serviceWorker.addEventListener('message', function(event) {
        if (event.data.type === 'CACHE_CLEARED') {
            showToast('Cache cleared from Service Worker', 'success');
        }
    });
}
</script>
@endsection