@extends('layouts.dashboard')

@section('title', 'Edit Client')

@section('content')
<div class="container mx-auto px-4 py-6">
    <!-- Header -->
    <div class="flex items-center justify-between mb-8">
        <div>
            <a href="{{ route('clients.index') }}" 
               class="text-primary-600 hover:text-primary-700 transition-colors duration-200 inline-flex items-center mb-2">
                <i class="fas fa-arrow-left mr-2"></i> Back to Clients
            </a>
            <h1 class="text-2xl font-bold text-gray-800">Edit Client</h1>
            <p class="text-gray-600 mt-1">Update client information</p>
        </div>
        <div class="flex items-center space-x-2">
            <div class="w-12 h-12 rounded-lg bg-gradient-to-r from-yellow-100 to-orange-200 flex items-center justify-center">
                <i class="fas fa-edit text-yellow-600 text-xl"></i>
            </div>
        </div>
    </div>

    <div class="max-w-4xl mx-auto">
        <div class="bg-white rounded-xl shadow-sm border border-gray-100 overflow-hidden">
            <!-- Form Header -->
            <div class="px-6 py-4 border-b border-gray-100 bg-gradient-to-r from-yellow-50 to-orange-100">
                <div class="flex items-center justify-between">
                    <div>
                        <h2 class="text-lg font-semibold text-gray-800">Edit {{ $client->name }}</h2>
                        <p class="text-sm text-gray-600 mt-1">Update the client details</p>
                    </div>
                    <span class="inline-flex items-center px-3 py-1 rounded-full text-sm font-medium 
                        {{ $client->status == 'active' ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800' }}">
                        <i class="fas fa-circle mr-1 text-{{ $client->status == 'active' ? 'green' : 'red' }}-500" style="font-size: 8px;"></i>
                        {{ ucfirst($client->status) }}
                    </span>
                </div>
            </div>

            <!-- Form -->
            <form action="{{ route('clients.update', $client) }}" method="POST" id="clientForm" class="p-6">
                @csrf
                @method('PUT')

                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <!-- Client Basic Information -->
                    <div class="space-y-6">
                        <h3 class="text-lg font-medium text-gray-800 border-b pb-2">Basic Information</h3>
                        
                        <!-- Client Name -->
                        <div>
                            <label for="name" class="form-label">
                                <i class="fas fa-handshake mr-2 text-gray-400"></i>Client Name *
                            </label>
                            <input type="text" 
                                   id="name" 
                                   name="name" 
                                   value="{{ old('name', $client->name) }}"
                                   class="form-input @error('name') border-red-500 @enderror"
                                   placeholder="e.g., Cairo Medical Center"
                                   required>
                            @error('name')
                                <p class="mt-1 text-sm text-red-600 flex items-center">
                                    <i class="fas fa-exclamation-circle mr-1"></i> {{ $message }}
                                </p>
                            @enderror
                        </div>

                        <!-- Contact Person -->
                        <div>
                            <label for="contact_person" class="form-label">
                                <i class="fas fa-user-tie mr-2 text-gray-400"></i>Contact Person
                            </label>
                            <input type="text" 
                                   id="contact_person" 
                                   name="contact_person" 
                                   value="{{ old('contact_person', $client->contact_person) }}"
                                   class="form-input @error('contact_person') border-red-500 @enderror"
                                   placeholder="e.g., Dr. Ahmed Ali">
                            @error('contact_person')
                                <p class="mt-1 text-sm text-red-600 flex items-center">
                                    <i class="fas fa-exclamation-circle mr-1"></i> {{ $message }}
                                </p>
                            @enderror
                        </div>

                        <!-- Phone -->
                        <div>
                            <label for="phone" class="form-label">
                                <i class="fas fa-phone mr-2 text-gray-400"></i>Phone Number *
                            </label>
                            <input type="tel" 
                                   id="phone" 
                                   name="phone" 
                                   value="{{ old('phone', $client->phone) }}"
                                   class="form-input @error('phone') border-red-500 @enderror"
                                   placeholder="e.g., +201234567890"
                                   required>
                            @error('phone')
                                <p class="mt-1 text-sm text-red-600 flex items-center">
                                    <i class="fas fa-exclamation-circle mr-1"></i> {{ $message }}
                                </p>
                            @enderror
                        </div>

                        <!-- Email -->
                        <div>
                            <label for="email" class="form-label">
                                <i class="fas fa-envelope mr-2 text-gray-400"></i>Email Address
                            </label>
                            <input type="email" 
                                   id="email" 
                                   name="email" 
                                   value="{{ old('email', $client->email) }}"
                                   class="form-input @error('email') border-red-500 @enderror"
                                   placeholder="e.g., info@clinic.com">
                            @error('email')
                                <p class="mt-1 text-sm text-red-600 flex items-center">
                                    <i class="fas fa-exclamation-circle mr-1"></i> {{ $message }}
                                </p>
                            @enderror
                        </div>

                        <!-- Status -->
                        <div>
                            <label for="status" class="form-label">
                                <i class="fas fa-power-off mr-2 text-gray-400"></i>Status *
                            </label>
                            <select id="status" 
                                    name="status" 
                                    class="form-select @error('status') border-red-500 @enderror"
                                    required>
                                <option value="active" {{ old('status', $client->status) == 'active' ? 'selected' : '' }}>Active</option>
                                <option value="inactive" {{ old('status', $client->status) == 'inactive' ? 'selected' : '' }}>Inactive</option>
                            </select>
                            @error('status')
                                <p class="mt-1 text-sm text-red-600 flex items-center">
                                    <i class="fas fa-exclamation-circle mr-1"></i> {{ $message }}
                                </p>
                            @enderror
                        </div>
                    </div>

                    <!-- Location Information -->
                    <div class="space-y-6">
                        <h3 class="text-lg font-medium text-gray-800 border-b pb-2">Location Information</h3>
                        
                        <!-- Governorate -->
                        <div>
                            <label for="governorate_id" class="form-label">
                                <i class="fas fa-flag mr-2 text-gray-400"></i>Governorate *
                            </label>
                            <select id="governorate_id" 
                                    name="governorate_id" 
                                    class="form-select @error('governorate_id') border-red-500 @enderror"
                                    required>
                                <option value="">Select Governorate</option>
                                @foreach($governorates as $governorate)
                                    <option value="{{ $governorate->id }}" {{ old('governorate_id', $client->governorate_id) == $governorate->id ? 'selected' : '' }}>
                                        {{ $governorate->name }}
                                    </option>
                                @endforeach
                            </select>
                            @error('governorate_id')
                                <p class="mt-1 text-sm text-red-600 flex items-center">
                                    <i class="fas fa-exclamation-circle mr-1"></i> {{ $message }}
                                </p>
                            @enderror
                        </div>

                        <!-- City -->
                        <div>
                            <label for="city_id" class="form-label">
                                <i class="fas fa-city mr-2 text-gray-400"></i>City *
                            </label>
                            <select id="city_id" 
                                    name="city_id" 
                                    class="form-select @error('city_id') border-red-500 @enderror"
                                    required>
                                <option value="">Select City</option>
                                @foreach($cities as $city)
                                    <option value="{{ $city->id }}" {{ old('city_id', $client->city_id) == $city->id ? 'selected' : '' }}>
                                        {{ $city->name }}
                                    </option>
                                @endforeach
                            </select>
                            @error('city_id')
                                <p class="mt-1 text-sm text-red-600 flex items-center">
                                    <i class="fas fa-exclamation-circle mr-1"></i> {{ $message }}
                                </p>
                            @enderror
                        </div>

                        <!-- Area -->
                        <div>
                            <label for="area" class="form-label">
                                <i class="fas fa-map-marker mr-2 text-gray-400"></i>Area *
                            </label>
                            <input type="text" 
                                   id="area" 
                                   name="area" 
                                   value="{{ old('area', $client->area) }}"
                                   class="form-input @error('area') border-red-500 @enderror"
                                   placeholder="e.g., Downtown, Nasr City"
                                   required>
                            @error('area')
                                <p class="mt-1 text-sm text-red-600 flex items-center">
                                    <i class="fas fa-exclamation-circle mr-1"></i> {{ $message }}
                                </p>
                            @enderror
                        </div>

                        <!-- Address -->
                        <div>
                            <label for="address" class="form-label">
                                <i class="fas fa-map-pin mr-2 text-gray-400"></i>Full Address
                            </label>
                            <textarea id="address" 
                                      name="address" 
                                      rows="3"
                                      class="form-input @error('address') border-red-500 @enderror"
                                      placeholder="Street name, building number, floor...">{{ old('address', $client->address) }}</textarea>
                            @error('address')
                                <p class="mt-1 text-sm text-red-600 flex items-center">
                                    <i class="fas fa-exclamation-circle mr-1"></i> {{ $message }}
                                </p>
                            @enderror
                        </div>
                    </div>
                </div>

                <!-- GPS Coordinates -->
                <div class="mt-8 pt-6 border-t border-gray-200">
                    <h3 class="text-lg font-medium text-gray-800 mb-4">GPS Coordinates (Optional)</h3>
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <!-- Latitude -->
                        <div>
                            <label for="latitude" class="form-label">
                                <i class="fas fa-globe-americas mr-2 text-gray-400"></i>Latitude
                            </label>
                            <div class="flex items-center space-x-3">
                                <input type="number" 
                                       id="latitude" 
                                       name="latitude" 
                                       value="{{ old('latitude', $client->latitude) }}"
                                       class="form-input @error('latitude') border-red-500 @enderror"
                                       placeholder="e.g., 30.0444"
                                       step="0.000001">
                                <button type="button" 
                                        id="getLocationBtn"
                                        class="btn-secondary whitespace-nowrap">
                                    <i class="fas fa-crosshairs mr-2"></i> Get Location
                                </button>
                            </div>
                            @error('latitude')
                                <p class="mt-1 text-sm text-red-600 flex items-center">
                                    <i class="fas fa-exclamation-circle mr-1"></i> {{ $message }}
                                </p>
                            @enderror
                        </div>

                        <!-- Longitude -->
                        <div>
                            <label for="longitude" class="form-label">
                                <i class="fas fa-globe-americas mr-2 text-gray-400"></i>Longitude
                            </label>
                            <input type="number" 
                                   id="longitude" 
                                   name="longitude" 
                                   value="{{ old('longitude', $client->longitude) }}"
                                   class="form-input @error('longitude') border-red-500 @enderror"
                                   placeholder="e.g., 31.2357"
                                   step="0.000001">
                            @error('longitude')
                                <p class="mt-1 text-sm text-red-600 flex items-center">
                                    <i class="fas fa-exclamation-circle mr-1"></i> {{ $message }}
                                </p>
                            @enderror
                        </div>
                    </div>
                    
                    <!-- Map Preview -->
                    <div id="mapPreview" class="mt-4 h-48 bg-gray-100 rounded-lg border border-gray-300">
                        <div id="map" class="w-full h-full rounded-lg"></div>
                    </div>
                </div>

                <!-- Form Footer -->
                <div class="mt-8 pt-6 border-t border-gray-100 flex items-center justify-between">
                    <div>
                        <p class="text-sm text-gray-500">
                            <i class="fas fa-info-circle mr-1"></i> Fields marked with * are required
                        </p>
                    </div>
                    <div class="flex items-center space-x-3">
                        <a href="{{ route('clients.index') }}" 
                           class="btn-secondary">
                            <i class="fas fa-times mr-2"></i> Cancel
                        </a>
                        <button type="submit" 
                                class="btn-primary">
                            <i class="fas fa-save mr-2"></i> Update Client
                        </button>
                    </div>
                </div>
            </form>
        </div>

        <!-- Danger Zone -->
        <div class="mt-6 bg-red-50 border border-red-100 rounded-lg p-4">
            <div class="flex">
                <div class="flex-shrink-0">
                    <i class="fas fa-exclamation-triangle text-red-500 text-lg"></i>
                </div>
                <div class="ml-3">
                    <h3 class="text-sm font-medium text-red-800">Danger Zone</h3>
                    <div class="mt-2">
                        <p class="text-sm text-red-700 mb-3">
                            Deleting this client will permanently remove it from the system. 
                            @if($client->visits()->count() > 0)
                            <span class="font-semibold">Cannot delete: This client has {{ $client->visits()->count() }} visits.</span>
                            @endif
                        </p>
                        @if($client->visits()->count() == 0)
                        <form action="{{ route('clients.destroy', $client) }}" 
                              method="POST" 
                              onsubmit="return confirmDelete()">
                            @csrf
                            @method('DELETE')
                            <button type="submit" 
                                    class="btn-danger inline-flex items-center">
                                <i class="fas fa-trash mr-2"></i> Delete Client
                            </button>
                        </form>
                        @else
                        <button type="button" 
                                class="btn-secondary cursor-not-allowed opacity-50 inline-flex items-center"
                                disabled>
                            <i class="fas fa-trash mr-2"></i> Delete Client (Disabled)
                        </button>
                        <p class="text-xs text-red-600 mt-2">
                            <i class="fas fa-info-circle mr-1"></i> Delete all visits for this client first
                        </p>
                        @endif
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Leaflet JS for Maps -->
<link rel="stylesheet" href="https://unpkg.com/leaflet@1.9.4/dist/leaflet.css" />
<script src="https://unpkg.com/leaflet@1.9.4/dist/leaflet.js"></script>

<script>
let map = null;
let marker = null;

document.addEventListener('DOMContentLoaded', function() {
    const governorateSelect = document.getElementById('governorate_id');
    const citySelect = document.getElementById('city_id');
    const latitudeInput = document.getElementById('latitude');
    const longitudeInput = document.getElementById('longitude');
    const getLocationBtn = document.getElementById('getLocationBtn');

    // Initialize map
    const clientLat = parseFloat("{{ $client->latitude ?? 30.0444 }}");
    const clientLng = parseFloat("{{ $client->longitude ?? 31.2357 }}");
    
    if (!isNaN(clientLat) && !isNaN(clientLng)) {
        initMap(clientLat, clientLng);
    }

    // Load cities based on selected governorate
    governorateSelect.addEventListener('change', function() {
        const governorateId = this.value;
        
        if (governorateId) {
            fetch(`/clients/cities?governorate_id=${governorateId}`)
                .then(response => response.json())
                .then(data => {
                    citySelect.innerHTML = '<option value="">Select City</option>';
                    data.forEach(city => {
                        const option = document.createElement('option');
                        option.value = city.id;
                        option.textContent = city.name;
                        option.selected = city.id == "{{ $client->city_id }}";
                        citySelect.appendChild(option);
                    });
                })
                .catch(error => console.error('Error fetching cities:', error));
        }
    });

    // Get current location
    getLocationBtn.addEventListener('click', function() {
        if (navigator.geolocation) {
            getLocationBtn.innerHTML = '<i class="fas fa-spinner fa-spin mr-2"></i> Getting Location...';
            getLocationBtn.disabled = true;
            
            navigator.geolocation.getCurrentPosition(
                function(position) {
                    latitudeInput.value = position.coords.latitude.toFixed(6);
                    longitudeInput.value = position.coords.longitude.toFixed(6);
                    getLocationBtn.innerHTML = '<i class="fas fa-crosshairs mr-2"></i> Get Location';
                    getLocationBtn.disabled = false;
                    updateMap();
                },
                function(error) {
                    alert('Unable to get your location. Please enter coordinates manually.');
                    getLocationBtn.innerHTML = '<i class="fas fa-crosshairs mr-2"></i> Get Location';
                    getLocationBtn.disabled = false;
                }
            );
        } else {
            alert('Geolocation is not supported by your browser.');
        }
    });

    // Update map when coordinates change
    latitudeInput.addEventListener('input', updateMap);
    longitudeInput.addEventListener('input', updateMap);

    function updateMap() {
        const lat = parseFloat(latitudeInput.value);
        const lng = parseFloat(longitudeInput.value);
        
        if (!isNaN(lat) && !isNaN(lng) && lat >= -90 && lat <= 90 && lng >= -180 && lng <= 180) {
            if (!map) {
                initMap(lat, lng);
            } else {
                map.setView([lat, lng], 15);
                if (marker) {
                    marker.setLatLng([lat, lng]);
                } else {
                    marker = L.marker([lat, lng]).addTo(map);
                }
            }
        }
    }

    function initMap(lat, lng) {
        map = L.map('map').setView([lat, lng], 15);
        
        L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
            attribution: '© OpenStreetMap contributors'
        }).addTo(map);
        
        marker = L.marker([lat, lng]).addTo(map);
    }
});

function confirmDelete() {
    return confirm('Are you sure you want to delete this client? This action cannot be undone.');
}
</script>

<style>
#map { 
    height: 100%; 
    width: 100%;
    border-radius: 0.5rem;
}
</style>
@endsection