@extends('layouts.dashboard')

@section('title', 'Employee Dashboard')

@section('content')
<div class="container mx-auto px-4 py-6">
    <!-- Welcome Message -->
    <div class="bg-gradient-to-r from-green-500 to-teal-600 rounded-xl shadow-md p-6 mb-6 text-white">
        <div class="flex items-center justify-between">
            <div>
                <h1 class="text-2xl font-bold mb-2">Welcome back, {{ auth()->user()->name }}!</h1>
                <p class="opacity-90">View your scheduled visits and manage your leave requests</p>
                <div class="mt-2 flex items-center">
                    <i class="fas fa-user-tie mr-2"></i>
                    <span>Supervisor: {{ auth()->user()->supervisor->name ?? 'Not assigned' }}</span>
                </div>
            </div>
            <div class="w-16 h-16 rounded-full bg-white/20 flex items-center justify-center">
                <i class="fas fa-user-md text-2xl"></i>
            </div>
        </div>
    </div>

    <!-- Employee Stats -->
    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
        <div class="card-hover bg-white rounded-xl shadow-sm border border-gray-100 p-6">
            <div class="flex items-center">
                <div class="p-3 rounded-full bg-blue-100 text-blue-600">
                    <i class="fas fa-map-marker-alt text-xl"></i>
                </div>
                <div class="ml-4">
                    <p class="text-sm font-medium text-gray-600">Total Visits</p>
                    <p class="text-2xl font-bold text-gray-800">{{ $stats['total_visits'] ?? 0 }}</p>
                </div>
            </div>
        </div>

        <div class="card-hover bg-white rounded-xl shadow-sm border border-gray-100 p-6">
            <div class="flex items-center">
                <div class="p-3 rounded-full bg-yellow-100 text-yellow-600">
                    <i class="fas fa-calendar-day text-xl"></i>
                </div>
                <div class="ml-4">
                    <p class="text-sm font-medium text-gray-600">Today's Visits</p>
                    <p class="text-2xl font-bold text-gray-800">{{ $stats['todays_visits'] ?? 0 }}</p>
                </div>
            </div>
        </div>

        <div class="card-hover bg-white rounded-xl shadow-sm border border-gray-100 p-6">
            <div class="flex items-center">
                <div class="p-3 rounded-full bg-green-100 text-green-600">
                    <i class="fas fa-check-circle text-xl"></i>
                </div>
                <div class="ml-4">
                    <p class="text-sm font-medium text-gray-600">Completed</p>
                    <p class="text-2xl font-bold text-gray-800">{{ $stats['completed_visits'] ?? 0 }}</p>
                </div>
            </div>
        </div>

        <div class="card-hover bg-white rounded-xl shadow-sm border border-gray-100 p-6">
            <div class="flex items-center">
                <div class="p-3 rounded-full bg-purple-100 text-purple-600">
                    <i class="fas fa-calendar-alt text-xl"></i>
                </div>
                <div class="ml-4">
                    <p class="text-sm font-medium text-gray-600">Leave Balance</p>
                    <p class="text-2xl font-bold text-gray-800">15 days</p>
                </div>
            </div>
        </div>
    </div>

    <!-- Two Column Layout -->
    <div class="grid grid-cols-1 lg:grid-cols-2 gap-6">
        <!-- Today's Schedule -->
        <div class="card-hover bg-white rounded-xl shadow-sm border border-gray-100">
            <div class="px-6 py-4 border-b border-gray-100 bg-gradient-to-r from-blue-50 to-blue-100">
                <h2 class="text-lg font-semibold text-gray-800">Today's Schedule</h2>
                <p class="text-sm text-gray-600">{{ now()->format('l, F j, Y') }}</p>
            </div>
            <div class="p-6">
                @php
                    $todaysVisits = auth()->user()->visits()
                        ->whereDate('scheduled_date', today())
                        ->with('client')
                        ->orderBy('scheduled_time')
                        ->get();
                @endphp
                
                @if($todaysVisits->count() > 0)
                    <div class="space-y-4">
                        @foreach($todaysVisits as $visit)
                        <div class="p-4 border border-gray-200 rounded-lg hover:bg-gray-50 transition duration-200">
                            <div class="flex items-start justify-between">
                                <div class="flex-1">
                                    <div class="flex items-center">
                                        <div class="w-10 h-10 rounded-full bg-gradient-to-r from-green-500 to-teal-600 flex items-center justify-center">
                                            <i class="fas fa-handshake text-white"></i>
                                        </div>
                                        <div class="ml-3">
                                            <div class="font-medium text-gray-900">{{ $visit->client->name }}</div>
                                            <div class="text-sm text-gray-500">{{ $visit->scheduled_time }} • {{ $visit->area }}</div>
                                        </div>
                                    </div>
                                    
                                    @if($visit->comments)
                                    <div class="mt-2 text-sm text-gray-600">
                                        <i class="fas fa-comment-alt mr-1 text-gray-400"></i>
                                        {{ Str::limit($visit->comments, 100) }}
                                    </div>
                                    @endif
                                </div>
                                
                                <div class="ml-4">
                                    @php
                                        $statusColors = [
                                            'scheduled' => 'bg-yellow-100 text-yellow-800',
                                            'in_progress' => 'bg-blue-100 text-blue-800',
                                            'completed' => 'bg-green-100 text-green-800',
                                            'cancelled' => 'bg-red-100 text-red-800'
                                        ];
                                    @endphp
                                    <span class="inline-flex items-center px-2 py-1 rounded text-xs font-medium {{ $statusColors[$visit->status] }}">
                                        {{ str_replace('_', ' ', ucfirst($visit->status)) }}
                                    </span>
                                    
                                    @if($visit->status == 'scheduled')
                                    <form action="{{ route('visits.start', $visit) }}" method="POST" class="mt-2">
                                        @csrf
                                        <button type="submit" 
                                                class="text-sm bg-green-600 hover:bg-green-700 text-white px-3 py-1 rounded flex items-center">
                                            <i class="fas fa-play-circle mr-1"></i> Start
                                        </button>
                                    </form>
                                    @endif
                                    
                                    @if($visit->status == 'in_progress')
                                    <button type="button" 
                                            onclick="showCompleteModal({{ $visit->id }})"
                                            class="mt-2 text-sm bg-purple-600 hover:bg-purple-700 text-white px-3 py-1 rounded flex items-center">
                                        <i class="fas fa-check-circle mr-1"></i> Complete
                                    </button>
                                    @endif
                                </div>
                            </div>
                            
                            @if($visit->latitude && $visit->longitude)
                            <div class="mt-3 flex items-center">
                                <i class="fas fa-map-marker-alt text-gray-400 mr-2"></i>
                                <a href="javascript:void(0)" 
                                   onclick="showVisitOnMap({{ $visit->latitude }}, {{ $visit->longitude }}, '{{ $visit->client->name }}')"
                                   class="text-sm text-blue-600 hover:text-blue-700">
                                    View on Map
                                </a>
                            </div>
                            @endif
                        </div>
                        @endforeach
                    </div>
                @else
                    <div class="text-center py-8">
                        <i class="fas fa-calendar-check text-gray-300 text-4xl mb-3"></i>
                        <p class="text-gray-500">No visits scheduled for today</p>
                        <p class="text-sm text-gray-400 mt-1">Enjoy your day!</p>
                    </div>
                @endif
                
                <div class="mt-4 text-center">
                    <a href="{{ route('visits.my-visits') }}" class="text-primary-600 hover:text-primary-700 font-medium">
                        View all my visits →
                    </a>
                </div>
            </div>
        </div>

        <!-- Quick Actions -->
        <div class="space-y-6">
            <!-- Upcoming Visits -->
            <div class="card-hover bg-white rounded-xl shadow-sm border border-gray-100">
                <div class="px-6 py-4 border-b border-gray-100">
                    <h2 class="text-lg font-semibold text-gray-800">Upcoming Visits</h2>
                </div>
                <div class="p-6">
                    @php
                        $upcomingVisits = auth()->user()->visits()
                            ->whereDate('scheduled_date', '>', today())
                            ->with('client')
                            ->orderBy('scheduled_date')
                            ->take(3)
                            ->get();
                    @endphp
                    
                    @if($upcomingVisits->count() > 0)
                        <div class="space-y-4">
                            @foreach($upcomingVisits as $visit)
                            <div class="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
                                <div>
                                    <div class="font-medium text-gray-900">{{ $visit->client->name }}</div>
                                    <div class="text-sm text-gray-500">
                                        {{ $visit->scheduled_date->format('M d') }} • {{ $visit->scheduled_time }}
                                    </div>
                                </div>
                                <div class="text-right">
                                    <span class="text-xs text-gray-500">{{ $visit->area }}</span>
                                </div>
                            </div>
                            @endforeach
                        </div>
                    @else
                        <div class="text-center py-4">
                            <i class="fas fa-calendar-plus text-gray-300 text-3xl mb-2"></i>
                            <p class="text-gray-500">No upcoming visits</p>
                        </div>
                    @endif
                </div>
            </div>

            <!-- Quick Actions -->
            <div class="card-hover bg-white rounded-xl shadow-sm border border-gray-100">
                <div class="px-6 py-4 border-b border-gray-100">
                    <h2 class="text-lg font-semibold text-gray-800">Quick Actions</h2>
                </div>
                <div class="p-6">
                    <div class="grid grid-cols-2 gap-4">
                        <a href="{{ route('visits.my-visits') }}" class="flex flex-col items-center p-4 border border-gray-200 rounded-lg hover:bg-gray-50 transition duration-200">
                            <div class="w-10 h-10 rounded-full bg-blue-100 flex items-center justify-center mb-2">
                                <i class="fas fa-calendar-alt text-blue-600"></i>
                            </div>
                            <span class="text-sm font-medium text-gray-900">My Visits</span>
                        </a>
                        
                        <a href="{{ route('leaves.create') }}" class="flex flex-col items-center p-4 border border-gray-200 rounded-lg hover:bg-gray-50 transition duration-200">
                            <div class="w-10 h-10 rounded-full bg-green-100 flex items-center justify-center mb-2">
                                <i class="fas fa-calendar-plus text-green-600"></i>
                            </div>
                            <span class="text-sm font-medium text-gray-900">Request Leave</span>
                        </a>
                        
                        <a href="{{ route('leaves.my-leaves') }}" class="flex flex-col items-center p-4 border border-gray-200 rounded-lg hover:bg-gray-50 transition duration-200">
                            <div class="w-10 h-10 rounded-full bg-yellow-100 flex items-center justify-center mb-2">
                                <i class="fas fa-calendar-check text-yellow-600"></i>
                            </div>
                            <span class="text-sm font-medium text-gray-900">My Leaves</span>
                        </a>
                        
                        <a href="{{ route('profile.edit') }}" class="flex flex-col items-center p-4 border border-gray-200 rounded-lg hover:bg-gray-50 transition duration-200">
                            <div class="w-10 h-10 rounded-full bg-purple-100 flex items-center justify-center mb-2">
                                <i class="fas fa-user-edit text-purple-600"></i>
                            </div>
                            <span class="text-sm font-medium text-gray-900">Profile</span>
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Complete Visit Modal -->
<div id="completeModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full hidden z-50">
    <div class="relative top-20 mx-auto p-5 border w-11/12 md:w-1/3 shadow-lg rounded-xl bg-white">
        <div class="flex justify-between items-center mb-4">
            <h3 class="text-lg font-semibold text-gray-800">Complete Visit</h3>
            <button onclick="closeCompleteModal()" class="text-gray-400 hover:text-gray-600">
                <i class="fas fa-times text-xl"></i>
            </button>
        </div>
        
        <form id="completeForm" method="POST">
            @csrf
            <div class="mb-4">
                <label class="form-label">Visit Notes</label>
                <textarea name="employee_notes" 
                          rows="4"
                          class="form-input"
                          placeholder="Add notes about the visit, client feedback, or any issues encountered..."></textarea>
            </div>
            
            <div class="flex justify-end space-x-3">
                <button type="button" 
                        onclick="closeCompleteModal()" 
                        class="btn-secondary">
                    Cancel
                </button>
                <button type="submit" 
                        class="btn-primary">
                    <i class="fas fa-check-circle mr-2"></i> Complete Visit
                </button>
            </div>
        </form>
    </div>
</div>

<!-- Map Modal -->
<div id="mapModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full hidden z-50">
    <div class="relative top-20 mx-auto p-5 border w-11/12 md:w-3/4 lg:w-1/2 shadow-lg rounded-xl bg-white">
        <div class="flex justify-between items-center mb-4">
            <h3 id="mapTitle" class="text-lg font-semibold text-gray-800">Visit Location</h3>
            <button onclick="closeMap()" class="text-gray-400 hover:text-gray-600">
                <i class="fas fa-times text-xl"></i>
            </button>
        </div>
        <div id="modalMap" class="h-96 rounded-lg border border-gray-300"></div>
        <div class="mt-4 flex justify-end">
            <button onclick="closeMap()" class="btn-secondary">
                Close
            </button>
        </div>
    </div>
</div>

<!-- Leaflet JS for Maps -->
<link rel="stylesheet" href="https://unpkg.com/leaflet@1.9.4/dist/leaflet.css" />
<script src="https://unpkg.com/leaflet@1.9.4/dist/leaflet.js"></script>

<script>
let modalMap = null;

function showCompleteModal(visitId) {
    const modal = document.getElementById('completeModal');
    const form = document.getElementById('completeForm');
    
    form.action = `/visits/${visitId}/complete`;
    modal.classList.remove('hidden');
}

function closeCompleteModal() {
    document.getElementById('completeModal').classList.add('hidden');
}

function showVisitOnMap(latitude, longitude, clientName) {
    const modal = document.getElementById('mapModal');
    const mapTitle = document.getElementById('mapTitle');
    const mapContainer = document.getElementById('modalMap');
    
    mapTitle.textContent = clientName + ' - Location';
    modal.classList.remove('hidden');
    
    // Clear previous map
    if (modalMap) {
        modalMap.remove();
    }
    
    // Initialize new map
    modalMap = L.map(mapContainer).setView([latitude, longitude], 15);
    
    L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
        attribution: '© OpenStreetMap contributors'
    }).addTo(modalMap);
    
    // Add marker for visit location
    L.marker([latitude, longitude])
        .addTo(modalMap)
        .bindPopup(`<b>${clientName}</b><br>Visit Location`)
        .openPopup();
}

function closeMap() {
    document.getElementById('mapModal').classList.add('hidden');
    if (modalMap) {
        modalMap.remove();
        modalMap = null;
    }
}
</script>

<style>
#modalMap { 
    height: 100%; 
    width: 100%;
    border-radius: 0.5rem;
}
</style>
@endsection