@extends('layouts.dashboard')

@section('title', 'Leave Management')

@section('content')
<div class="container mx-auto px-4 py-6">
    <!-- Header -->
    <div class="flex flex-col md:flex-row md:items-center md:justify-between mb-6">
        <div>
            <h1 class="text-2xl font-bold text-gray-800">Leave Management</h1>
            <p class="text-gray-600 mt-1">Manage employee leave requests and approvals</p>
        </div>
        <div class="flex items-center space-x-3 mt-4 md:mt-0">
            <div class="relative">
                <input type="search" 
                       placeholder="Search leaves..." 
                       class="pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-primary-500 focus:border-transparent">
                <i class="fas fa-search absolute left-3 top-3 text-gray-400"></i>
            </div>
        </div>
    </div>

    <!-- Stats Cards -->
    <div class="grid grid-cols-1 md:grid-cols-4 gap-6 mb-8">
        <div class="card-hover bg-white rounded-xl shadow-sm border border-gray-100 p-6">
            <div class="flex items-center">
                <div class="p-3 rounded-full bg-yellow-100 text-yellow-600">
                    <i class="fas fa-clock text-xl"></i>
                </div>
                <div class="ml-4">
                    <p class="text-sm font-medium text-gray-600">Pending</p>
                    <p class="text-2xl font-bold text-gray-800">
                        {{ \App\Models\LeaveRequest::where('status', 'pending')->count() }}
                    </p>
                </div>
            </div>
        </div>

        <div class="card-hover bg-white rounded-xl shadow-sm border border-gray-100 p-6">
            <div class="flex items-center">
                <div class="p-3 rounded-full bg-green-100 text-green-600">
                    <i class="fas fa-check-circle text-xl"></i>
                </div>
                <div class="ml-4">
                    <p class="text-sm font-medium text-gray-600">Approved</p>
                    <p class="text-2xl font-bold text-gray-800">
                        {{ \App\Models\LeaveRequest::where('status', 'approved')->count() }}
                    </p>
                </div>
            </div>
        </div>

        <div class="card-hover bg-white rounded-xl shadow-sm border border-gray-100 p-6">
            <div class="flex items-center">
                <div class="p-3 rounded-full bg-red-100 text-red-600">
                    <i class="fas fa-times-circle text-xl"></i>
                </div>
                <div class="ml-4">
                    <p class="text-sm font-medium text-gray-600">Rejected</p>
                    <p class="text-2xl font-bold text-gray-800">
                        {{ \App\Models\LeaveRequest::where('status', 'rejected')->count() }}
                    </p>
                </div>
            </div>
        </div>

        <div class="card-hover bg-white rounded-xl shadow-sm border border-gray-100 p-6">
            <div class="flex items-center">
                <div class="p-3 rounded-full bg-blue-100 text-blue-600">
                    <i class="fas fa-calendar-alt text-xl"></i>
                </div>
                <div class="ml-4">
                    <p class="text-sm font-medium text-gray-600">This Month</p>
                    <p class="text-2xl font-bold text-gray-800">
                        {{ \App\Models\LeaveRequest::whereMonth('created_at', now()->month)->count() }}
                    </p>
                </div>
            </div>
        </div>
    </div>

    <!-- Filters -->
    <div class="bg-white rounded-xl shadow-sm border border-gray-100 p-4 mb-6">
        <div class="flex flex-wrap gap-4">
            <select class="form-select w-full md:w-auto" id="statusFilter">
                <option value="">All Status</option>
                <option value="pending">Pending</option>
                <option value="approved">Approved</option>
                <option value="rejected">Rejected</option>
            </select>
            
            <select class="form-select w-full md:w-auto" id="typeFilter">
                <option value="">All Types</option>
                <option value="sick">Sick Leave</option>
                <option value="vacation">Vacation</option>
                <option value="personal">Personal</option>
                <option value="emergency">Emergency</option>
            </select>
            
            <input type="date" class="form-input w-full md:w-auto" id="fromDate" placeholder="From Date">
            
            <input type="date" class="form-input w-full md:w-auto" id="toDate" placeholder="To Date">
            
            <button class="btn-secondary" onclick="applyFilters()">
                <i class="fas fa-filter mr-2"></i> Filter
            </button>
            
            <button class="btn-secondary" onclick="resetFilters()">
                <i class="fas fa-redo mr-2"></i> Reset
            </button>
        </div>
    </div>

    <!-- Leaves Table -->
    <div class="bg-white rounded-xl shadow-sm border border-gray-100 overflow-hidden">
        <div class="overflow-x-auto">
            <table class="min-w-full divide-y divide-gray-200">
                <thead class="bg-gray-50">
                    <tr>
                        <th class="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase tracking-wider">
                            <div class="flex items-center">
                                <i class="fas fa-user mr-2"></i> Employee
                            </div>
                        </th>
                        <th class="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase tracking-wider">
                            <div class="flex items-center">
                                <i class="fas fa-calendar mr-2"></i> Leave Period
                            </div>
                        </th>
                        <th class="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase tracking-wider">
                            <div class="flex items-center">
                                <i class="fas fa-info-circle mr-2"></i> Type & Reason
                            </div>
                        </th>
                        <th class="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase tracking-wider">
                            <div class="flex items-center">
                                <i class="fas fa-clock mr-2"></i> Duration
                            </div>
                        </th>
                        <th class="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase tracking-wider">
                            <div class="flex items-center">
                                <i class="fas fa-check-circle mr-2"></i> Status
                            </div>
                        </th>
                        <th class="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase tracking-wider">
                            <div class="flex items-center">
                                <i class="fas fa-cog mr-2"></i> Actions
                            </div>
                        </th>
                    </tr>
                </thead>
                <tbody class="bg-white divide-y divide-gray-100">
                    @forelse($leaves as $leave)
                    <tr class="hover:bg-gray-50 transition-colors duration-150">
                        <td class="px-6 py-4 whitespace-nowrap">
                            <div class="flex items-center">
                                <div class="flex-shrink-0 h-10 w-10">
                                    <div class="h-10 w-10 rounded-full bg-gradient-to-r from-blue-400 to-blue-500 flex items-center justify-center">
                                        <i class="fas fa-user text-white"></i>
                                    </div>
                                </div>
                                <div class="ml-4">
                                    <div class="text-sm font-semibold text-gray-900">{{ $leave->employee->name }}</div>
                                    <div class="text-sm text-gray-500">{{ $leave->employee->employee_id }}</div>
                                    <div class="text-xs text-gray-400">Dept: {{ $leave->employee->department->name ?? 'N/A' }}</div>
                                </div>
                            </div>
                        </td>
                        <td class="px-6 py-4">
                            <div class="text-sm text-gray-900">
                                <div class="flex items-center">
                                    <i class="fas fa-calendar-day text-gray-400 mr-2"></i>
                                    <span>{{ $leave->start_date->format('M d, Y') }}</span>
                                </div>
                                <div class="flex items-center mt-1">
                                    <i class="fas fa-calendar-alt text-gray-400 mr-2"></i>
                                    <span>{{ $leave->end_date->format('M d, Y') }}</span>
                                </div>
                            </div>
                        </td>
                        <td class="px-6 py-4">
                            <div class="text-sm text-gray-900">
                                @php
                                    $typeColors = [
                                        'sick' => 'bg-red-100 text-red-800',
                                        'vacation' => 'bg-blue-100 text-blue-800',
                                        'personal' => 'bg-yellow-100 text-yellow-800',
                                        'emergency' => 'bg-purple-100 text-purple-800'
                                    ];
                                @endphp
                                <span class="inline-flex items-center px-2 py-1 rounded text-xs font-medium {{ $typeColors[$leave->type] }} mb-2">
                                    {{ ucfirst($leave->type) }} Leave
                                </span>
                                <div class="text-sm text-gray-600 mt-1">
                                    {{ Str::limit($leave->reason, 100) }}
                                </div>
                            </div>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap">
                            <div class="text-center">
                                <div class="text-2xl font-bold text-gray-800">{{ $leave->days }}</div>
                                <div class="text-sm text-gray-500">days</div>
                            </div>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap">
                            @php
                                $statusColors = [
                                    'pending' => 'bg-yellow-100 text-yellow-800',
                                    'approved' => 'bg-green-100 text-green-800',
                                    'rejected' => 'bg-red-100 text-red-800'
                                ];
                            @endphp
                            <span class="inline-flex items-center px-3 py-1 rounded-full text-sm font-medium {{ $statusColors[$leave->status] }}">
                                <i class="fas fa-circle mr-1 text-{{ 
                                    $leave->status == 'pending' ? 'yellow' : 
                                    ($leave->status == 'approved' ? 'green' : 'red') 
                                }}-500" style="font-size: 8px;"></i>
                                {{ ucfirst($leave->status) }}
                            </span>
                            @if($leave->approved_by)
                            <div class="text-xs text-gray-500 mt-1">
                                By: {{ $leave->approvedBy->name ?? 'Admin' }}
                            </div>
                            @endif
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                            <div class="flex items-center space-x-2">
                                @if($leave->status == 'pending')
                                    @can('approve', $leave)
                                    <button type="button" 
                                            onclick="showApproveModal({{ $leave->id }})"
                                            class="text-green-600 hover:text-green-900 transition-colors duration-200"
                                            title="Approve Leave">
                                        <i class="fas fa-check"></i>
                                    </button>
                                    <button type="button" 
                                            onclick="showRejectModal({{ $leave->id }})"
                                            class="text-red-600 hover:text-red-900 transition-colors duration-200"
                                            title="Reject Leave">
                                        <i class="fas fa-times"></i>
                                    </button>
                                    @endcan
                                @endif
                                
                                @if($leave->status == 'pending' && auth()->user()->id == $leave->employee_id)
                                <form action="{{ route('leaves.destroy', $leave) }}" 
                                      method="POST" 
                                      class="inline"
                                      onsubmit="return confirmDelete()">
                                    @csrf
                                    @method('DELETE')
                                    <button type="submit" 
                                            class="text-red-600 hover:text-red-900 transition-colors duration-200"
                                            title="Delete Leave Request">
                                        <i class="fas fa-trash"></i>
                                    </button>
                                </form>
                                @endif
                            </div>
                        </td>
                    </tr>
                    @empty
                    <tr>
                        <td colspan="6" class="px-6 py-12 text-center">
                            <div class="flex flex-col items-center justify-center">
                                <div class="w-24 h-24 mb-4 text-gray-300">
                                    <i class="fas fa-calendar-times text-6xl"></i>
                                </div>
                                <h3 class="text-lg font-medium text-gray-700 mb-2">No Leave Requests</h3>
                                <p class="text-gray-500">No leave requests found.</p>
                            </div>
                        </td>
                    </tr>
                    @endforelse
                </tbody>
            </table>
        </div>

        <!-- Pagination -->
        @if($leaves->hasPages())
        <div class="bg-gray-50 px-6 py-4 border-t border-gray-100">
            {{ $leaves->links() }}
        </div>
        @endif
    </div>
</div>

<!-- Approve Modal -->
<div id="approveModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full hidden z-50">
    <div class="relative top-20 mx-auto p-5 border w-11/12 md:w-1/3 shadow-lg rounded-xl bg-white">
        <div class="flex justify-between items-center mb-4">
            <h3 class="text-lg font-semibold text-gray-800">Approve Leave Request</h3>
            <button onclick="closeApproveModal()" class="text-gray-400 hover:text-gray-600">
                <i class="fas fa-times text-xl"></i>
            </button>
        </div>
        
        <form id="approveForm" method="POST">
            @csrf
            <div class="mb-4">
                <label class="form-label">Approval Notes (Optional)</label>
                <textarea name="admin_notes" 
                          rows="4"
                          class="form-input"
                          placeholder="Add any notes or instructions..."></textarea>
            </div>
            
            <div class="flex justify-end space-x-3">
                <button type="button" 
                        onclick="closeApproveModal()" 
                        class="btn-secondary">
                    Cancel
                </button>
                <button type="submit" 
                        class="btn-primary">
                    <i class="fas fa-check-circle mr-2"></i> Approve Leave
                </button>
            </div>
        </form>
    </div>
</div>

<!-- Reject Modal -->
<div id="rejectModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full hidden z-50">
    <div class="relative top-20 mx-auto p-5 border w-11/12 md:w-1/3 shadow-lg rounded-xl bg-white">
        <div class="flex justify-between items-center mb-4">
            <h3 class="text-lg font-semibold text-gray-800">Reject Leave Request</h3>
            <button onclick="closeRejectModal()" class="text-gray-400 hover:text-gray-600">
                <i class="fas fa-times text-xl"></i>
            </button>
        </div>
        
        <form id="rejectForm" method="POST">
            @csrf
            <div class="mb-4">
                <label class="form-label">Rejection Reason *</label>
                <textarea name="admin_notes" 
                          rows="4"
                          class="form-input"
                          placeholder="Please provide a reason for rejection..."
                          required></textarea>
            </div>
            
            <div class="flex justify-end space-x-3">
                <button type="button" 
                        onclick="closeRejectModal()" 
                        class="btn-secondary">
                    Cancel
                </button>
                <button type="submit" 
                        class="btn-danger">
                    <i class="fas fa-times-circle mr-2"></i> Reject Leave
                </button>
            </div>
        </form>
    </div>
</div>

<script>
function confirmDelete() {
    return confirm('Are you sure you want to delete this leave request? This action cannot be undone.');
}

function showApproveModal(leaveId) {
    const modal = document.getElementById('approveModal');
    const form = document.getElementById('approveForm');
    
    form.action = `/leaves/${leaveId}/approve`;
    modal.classList.remove('hidden');
}

function closeApproveModal() {
    document.getElementById('approveModal').classList.add('hidden');
}

function showRejectModal(leaveId) {
    const modal = document.getElementById('rejectModal');
    const form = document.getElementById('rejectForm');
    
    form.action = `/leaves/${leaveId}/reject`;
    modal.classList.remove('hidden');
}

function closeRejectModal() {
    document.getElementById('rejectModal').classList.add('hidden');
}

function applyFilters() {
    const status = document.getElementById('statusFilter').value;
    const type = document.getElementById('typeFilter').value;
    const fromDate = document.getElementById('fromDate').value;
    const toDate = document.getElementById('toDate').value;
    
    let url = new URL(window.location.href);
    let params = new URLSearchParams(url.search);
    
    if (status) params.set('status', status);
    if (type) params.set('type', type);
    if (fromDate) params.set('from_date', fromDate);
    if (toDate) params.set('to_date', toDate);
    
    window.location.href = url.pathname + '?' + params.toString();
}

function resetFilters() {
    window.location.href = window.location.pathname;
}
</script>
@endsection