@extends('layouts.dashboard')

@section('title', 'Payroll Details')

@section('content')
<div class="container mx-auto px-4 py-6">
    <div class="max-w-6xl mx-auto">
        <!-- Header -->
        <div class="mb-8">
            <div class="flex flex-col md:flex-row md:items-center md:justify-between">
                <div>
                    <h1 class="text-2xl font-bold text-gray-800">Payroll Details</h1>
                    <p class="text-gray-600 mt-1">View payroll information for {{ $payroll->employee->name }}</p>
                </div>
                <div class="flex items-center space-x-3 mt-4 md:mt-0">
                    <a href="{{ route('payrolls.index') }}" 
                       class="btn-secondary inline-flex items-center">
                        <i class="fas fa-arrow-left mr-2"></i> Back to Payrolls
                    </a>
                    
                    <!-- Status Actions -->
                    @if($payroll->status == 'draft')
                    <a href="{{ route('payrolls.edit', $payroll) }}" 
                       class="btn-primary inline-flex items-center">
                        <i class="fas fa-edit mr-2"></i> Edit
                    </a>
                    @endif
                    
                    @if(auth()->user()->can('admin'))
                    <div class="relative" x-data="{ open: false }">
                        <button @click="open = !open" 
                                class="btn-primary inline-flex items-center">
                            <i class="fas fa-cog mr-2"></i> Actions
                        </button>
                        
                        <div x-show="open" 
                             @click.away="open = false"
                             class="absolute right-0 mt-2 w-48 bg-white rounded-lg shadow-lg border border-gray-200 py-1 z-10">
                            @if($payroll->status == 'draft')
                            <form action="{{ route('payrolls.process', $payroll) }}" method="POST">
                                @csrf
                                <button type="submit" 
                                        class="block w-full text-left px-4 py-2 text-sm text-gray-700 hover:bg-gray-100"
                                        onclick="return confirm('Process this payroll?')">
                                    <i class="fas fa-check mr-2 text-green-500"></i> Process Payroll
                                </button>
                            </form>
                            @endif
                            
                            @if($payroll->status == 'processed')
                            <form action="{{ route('payrolls.mark-paid', $payroll) }}" method="POST">
                                @csrf
                                <button type="submit" 
                                        class="block w-full text-left px-4 py-2 text-sm text-gray-700 hover:bg-gray-100"
                                        onclick="return confirm('Mark this payroll as paid?')">
                                    <i class="fas fa-money-check-alt mr-2 text-purple-500"></i> Mark as Paid
                                </button>
                            </form>
                            @endif
                            
                            @if($payroll->status == 'draft')
                            <form action="{{ route('payrolls.destroy', $payroll) }}" 
                                  method="POST" 
                                  onsubmit="return confirm('Are you sure you want to delete this payroll?')">
                                @csrf
                                @method('DELETE')
                                <button type="submit" 
                                        class="block w-full text-left px-4 py-2 text-sm text-red-600 hover:bg-red-50">
                                    <i class="fas fa-trash mr-2"></i> Delete Payroll
                                </button>
                            </form>
                            @endif
                        </div>
                    </div>
                    @endif
                </div>
            </div>
        </div>

        <!-- Status Banner -->
        @php
            $statusColors = [
                'draft' => 'bg-yellow-50 border-yellow-200 text-yellow-800',
                'processed' => 'bg-blue-50 border-blue-200 text-blue-800',
                'paid' => 'bg-green-50 border-green-200 text-green-800'
            ];
            $statusIcons = [
                'draft' => 'clock',
                'processed' => 'check-circle',
                'paid' => 'money-check-alt'
            ];
        @endphp
        
        <div class="mb-6 {{ $statusColors[$payroll->status] }} border rounded-lg p-4">
            <div class="flex items-center justify-between">
                <div class="flex items-center">
                    <i class="fas fa-{{ $statusIcons[$payroll->status] }} text-xl mr-3"></i>
                    <div>
                        <h3 class="font-semibold">Payroll Status: {{ ucfirst($payroll->status) }}</h3>
                        <p class="text-sm opacity-90">
                            @if($payroll->status == 'draft')
                            This payroll is in draft status and can be edited or processed.
                            @elseif($payroll->status == 'processed')
                            This payroll has been processed and is ready for payment.
                            @else
                            This payroll has been paid on {{ $payroll->payment_date->format('F d, Y') }}.
                            @endif
                        </p>
                    </div>
                </div>
                <span class="px-3 py-1 rounded-full text-sm font-semibold {{ $statusColors[$payroll->status] }}">
                    {{ strtoupper($payroll->status) }}
                </span>
            </div>
        </div>

        <!-- Main Content Grid -->
        <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
            <!-- Left Column - Employee & Pay Period Info -->
            <div class="lg:col-span-2 space-y-6">
                <!-- Employee Card -->
                <div class="bg-white rounded-xl shadow-sm border border-gray-100 overflow-hidden">
                    <div class="p-6">
                        <h3 class="text-lg font-semibold text-gray-800 mb-4 flex items-center">
                            <i class="fas fa-user-tie mr-2"></i> Employee Information
                        </h3>
                        
                        <div class="flex items-start space-x-4">
                            <div class="flex-shrink-0">
                                <div class="w-20 h-20 rounded-full bg-gradient-to-r from-blue-400 to-blue-500 flex items-center justify-center">
                                    <i class="fas fa-user text-white text-2xl"></i>
                                </div>
                            </div>
                            <div class="flex-1">
                                <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                                    <div>
                                        <p class="text-sm text-gray-600">Full Name</p>
                                        <p class="font-semibold text-gray-800">{{ $payroll->employee->name }}</p>
                                    </div>
                                    <div>
                                        <p class="text-sm text-gray-600">Employee ID</p>
                                        <p class="font-semibold text-gray-800">{{ $payroll->employee->employee_id ?? 'N/A' }}</p>
                                    </div>
                                    <div>
                                        <p class="text-sm text-gray-600">Department</p>
                                        <p class="font-semibold text-gray-800">{{ $payroll->employee->department->name ?? 'N/A' }}</p>
                                    </div>
                                    <div>
                                        <p class="text-sm text-gray-600">Email</p>
                                        <p class="font-semibold text-gray-800">{{ $payroll->employee->email }}</p>
                                    </div>
                                    <div>
                                        <p class="text-sm text-gray-600">Phone</p>
                                        <p class="font-semibold text-gray-800">{{ $payroll->employee->phone ?? 'N/A' }}</p>
                                    </div>
                                    <div>
                                        <p class="text-sm text-gray-600">Employment Date</p>
                                        <p class="font-semibold text-gray-800">
                                            {{ $payroll->employee->hire_date ? $payroll->employee->hire_date->format('M d, Y') : 'N/A' }}
                                        </p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Pay Period Card -->
                <div class="bg-white rounded-xl shadow-sm border border-gray-100 overflow-hidden">
                    <div class="p-6">
                        <h3 class="text-lg font-semibold text-gray-800 mb-4 flex items-center">
                            <i class="fas fa-calendar-alt mr-2"></i> Pay Period & Payment Details
                        </h3>
                        
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                            <div>
                                <p class="text-sm text-gray-600">Pay Period</p>
                                <div class="flex items-center space-x-2 mt-1">
                                    <span class="font-semibold text-gray-800">{{ $payroll->pay_period_start->format('M d, Y') }}</span>
                                    <i class="fas fa-arrow-right text-gray-400"></i>
                                    <span class="font-semibold text-gray-800">{{ $payroll->pay_period_end->format('M d, Y') }}</span>
                                </div>
                                <p class="text-xs text-gray-500 mt-1">
                                    {{ $payroll->pay_period_start->diffInDays($payroll->pay_period_end) + 1 }} days
                                </p>
                            </div>
                            
                            <div>
                                <p class="text-sm text-gray-600">Payment Date</p>
                                <div class="flex items-center mt-1">
                                    @if($payroll->payment_date)
                                    <i class="fas fa-calendar-check text-green-500 mr-2"></i>
                                    <span class="font-semibold text-gray-800">{{ $payroll->payment_date->format('F d, Y') }}</span>
                                    @else
                                    <i class="fas fa-calendar-times text-gray-400 mr-2"></i>
                                    <span class="font-semibold text-gray-800">Not scheduled</span>
                                    @endif
                                </div>
                            </div>
                            
                            <div>
                                <p class="text-sm text-gray-600">Created At</p>
                                <div class="flex items-center mt-1">
                                    <i class="fas fa-clock text-gray-400 mr-2"></i>
                                    <span class="font-semibold text-gray-800">{{ $payroll->created_at->format('M d, Y h:i A') }}</span>
                                </div>
                            </div>
                            
                            <div>
                                <p class="text-sm text-gray-600">Last Updated</p>
                                <div class="flex items-center mt-1">
                                    <i class="fas fa-sync text-gray-400 mr-2"></i>
                                    <span class="font-semibold text-gray-800">{{ $payroll->updated_at->format('M d, Y h:i A') }}</span>
                                </div>
                            </div>
                            
                            @if($payroll->processed_at)
                            <div>
                                <p class="text-sm text-gray-600">Processed At</p>
                                <div class="flex items-center mt-1">
                                    <i class="fas fa-check-circle text-blue-500 mr-2"></i>
                                    <span class="font-semibold text-gray-800">{{ $payroll->processed_at->format('M d, Y h:i A') }}</span>
                                </div>
                            </div>
                            @endif
                        </div>
                    </div>
                </div>

                <!-- Notes Card -->
                @if($payroll->notes)
                <div class="bg-white rounded-xl shadow-sm border border-gray-100 overflow-hidden">
                    <div class="p-6">
                        <h3 class="text-lg font-semibold text-gray-800 mb-4 flex items-center">
                            <i class="fas fa-sticky-note mr-2"></i> Notes
                        </h3>
                        <div class="prose max-w-none">
                            <p class="text-gray-700 whitespace-pre-line">{{ $payroll->notes }}</p>
                        </div>
                    </div>
                </div>
                @endif
            </div>

            <!-- Right Column - Salary Breakdown -->
            <div class="space-y-6">
                <!-- Salary Summary Card -->
                <div class="bg-white rounded-xl shadow-sm border border-gray-100 overflow-hidden">
                    <div class="p-6">
                        <h3 class="text-lg font-semibold text-gray-800 mb-6 flex items-center justify-between">
                            <span class="flex items-center">
                                <i class="fas fa-money-bill-wave mr-2"></i> Salary Summary
                            </span>
                            <span class="text-sm font-normal text-gray-500">EGP</span>
                        </h3>
                        
                        <!-- Earnings -->
                        <div class="mb-6">
                            <h4 class="text-sm font-semibold text-gray-700 mb-3 flex items-center">
                                <i class="fas fa-arrow-up text-green-500 mr-2"></i> Earnings
                            </h4>
                            <div class="space-y-2">
                                <div class="flex justify-between items-center">
                                    <span class="text-gray-600">Basic Salary</span>
                                    <span class="font-medium">{{ number_format($payroll->basic_salary, 2) }}</span>
                                </div>
                                
                                @if($payroll->allowances > 0)
                                <div class="flex justify-between items-center">
                                    <span class="text-gray-600">Allowances</span>
                                    <span class="font-medium text-green-600">+ {{ number_format($payroll->allowances, 2) }}</span>
                                </div>
                                @endif
                                
                                @if($payroll->overtime > 0)
                                <div class="flex justify-between items-center">
                                    <span class="text-gray-600">Overtime</span>
                                    <span class="font-medium text-green-600">+ {{ number_format($payroll->overtime, 2) }}</span>
                                </div>
                                @endif
                                
                                @if($payroll->bonus > 0)
                                <div class="flex justify-between items-center">
                                    <span class="text-gray-600">Bonus</span>
                                    <span class="font-medium text-green-600">+ {{ number_format($payroll->bonus, 2) }}</span>
                                </div>
                                @endif
                            </div>
                        </div>
                        
                        <!-- Deductions -->
                        @if($payroll->deductions > 0)
                        <div class="mb-6 pt-4 border-t border-gray-200">
                            <h4 class="text-sm font-semibold text-gray-700 mb-3 flex items-center">
                                <i class="fas fa-arrow-down text-red-500 mr-2"></i> Deductions
                            </h4>
                            <div class="flex justify-between items-center">
                                <span class="text-gray-600">Total Deductions</span>
                                <span class="font-medium text-red-600">- {{ number_format($payroll->deductions, 2) }}</span>
                            </div>
                        </div>
                        @endif
                        
                        <!-- Net Salary -->
                        <div class="pt-4 border-t border-gray-200">
                            <div class="flex justify-between items-center mb-2">
                                <span class="font-semibold text-gray-700">Total Earnings</span>
                                <span class="font-semibold">
                                    {{ number_format($payroll->basic_salary + $payroll->allowances + $payroll->overtime + $payroll->bonus, 2) }}
                                </span>
                            </div>
                            <div class="flex justify-between items-center text-sm text-gray-500 mb-4">
                                <span>Before deductions</span>
                                <span>EGP</span>
                            </div>
                            
                            <div class="bg-gradient-to-r from-blue-50 to-blue-100 rounded-lg p-4">
                                <div class="flex justify-between items-center">
                                    <div>
                                        <p class="text-sm text-gray-600">Net Salary</p>
                                        <p class="text-2xl font-bold text-blue-600">{{ number_format($payroll->net_salary, 2) }} EGP</p>
                                    </div>
                                    <div class="text-blue-500">
                                        <i class="fas fa-hand-holding-usd text-3xl"></i>
                                    </div>
                                </div>
                                <p class="text-xs text-gray-500 mt-2">
                                    Amount to be paid to employee
                                </p>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Quick Stats Card -->
                <div class="bg-white rounded-xl shadow-sm border border-gray-100 overflow-hidden">
                    <div class="p-6">
                        <h3 class="text-lg font-semibold text-gray-800 mb-4 flex items-center">
                            <i class="fas fa-chart-pie mr-2"></i> Salary Breakdown
                        </h3>
                        
                        <div class="space-y-4">
                            <!-- Basic Salary Percentage -->
                            <div>
                                <div class="flex justify-between text-sm mb-1">
                                    <span class="text-gray-600">Basic Salary</span>
                                    <span class="font-medium">{{ number_format(($payroll->basic_salary / $payroll->net_salary) * 100, 1) }}%</span>
                                </div>
                                <div class="w-full bg-gray-200 rounded-full h-2">
                                    <div class="bg-blue-500 h-2 rounded-full" 
                                         style="width: {{ min(100, ($payroll->basic_salary / $payroll->net_salary) * 100) }}%"></div>
                                </div>
                            </div>
                            
                            <!-- Additional Earnings Percentage -->
                            @php
                                $additionalEarnings = $payroll->allowances + $payroll->overtime + $payroll->bonus;
                                $additionalPercentage = $additionalEarnings > 0 ? ($additionalEarnings / $payroll->net_salary) * 100 : 0;
                            @endphp
                            @if($additionalEarnings > 0)
                            <div>
                                <div class="flex justify-between text-sm mb-1">
                                    <span class="text-gray-600">Additional Earnings</span>
                                    <span class="font-medium">{{ number_format($additionalPercentage, 1) }}%</span>
                                </div>
                                <div class="w-full bg-gray-200 rounded-full h-2">
                                    <div class="bg-green-500 h-2 rounded-full" 
                                         style="width: {{ min(100, $additionalPercentage) }}%"></div>
                                </div>
                            </div>
                            @endif
                            
                            <!-- Deductions Percentage -->
                            @if($payroll->deductions > 0)
                            <div>
                                <div class="flex justify-between text-sm mb-1">
                                    <span class="text-gray-600">Deductions</span>
                                    <span class="font-medium">{{ number_format(($payroll->deductions / $payroll->net_salary) * 100, 1) }}%</span>
                                </div>
                                <div class="w-full bg-gray-200 rounded-full h-2">
                                    <div class="bg-red-500 h-2 rounded-full" 
                                         style="width: {{ min(100, ($payroll->deductions / $payroll->net_salary) * 100) }}%"></div>
                                </div>
                            </div>
                            @endif
                        </div>
                        
                        <div class="mt-6 pt-4 border-t border-gray-200">
                            <div class="text-center">
                                <p class="text-sm text-gray-600">Monthly Comparison</p>
                                <div class="flex items-center justify-center mt-2">
                                    @if($payroll->net_salary > 5000)
                                    <i class="fas fa-arrow-up text-green-500 mr-1"></i>
                                    <span class="text-green-600 font-medium">Above average</span>
                                    @elseif($payroll->net_salary < 5000)
                                    <i class="fas fa-arrow-down text-red-500 mr-1"></i>
                                    <span class="text-red-600 font-medium">Below average</span>
                                    @else
                                    <i class="fas fa-equals text-gray-500 mr-1"></i>
                                    <span class="text-gray-600 font-medium">Average</span>
                                    @endif
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Actions Card -->
                <div class="bg-white rounded-xl shadow-sm border border-gray-100 overflow-hidden">
                    <div class="p-6">
                        <h3 class="text-lg font-semibold text-gray-800 mb-4 flex items-center">
                            <i class="fas fa-bolt mr-2"></i> Quick Actions
                        </h3>
                        
                        <div class="space-y-3">
                            <a href="{{ route('payrolls.index') }}" 
                               class="flex items-center justify-between p-3 rounded-lg border border-gray-200 hover:bg-gray-50 transition-colors duration-200">
                                <div class="flex items-center">
                                    <div class="w-10 h-10 rounded-lg bg-blue-100 flex items-center justify-center mr-3">
                                        <i class="fas fa-list text-blue-600"></i>
                                    </div>
                                    <span class="font-medium">All Payrolls</span>
                                </div>
                                <i class="fas fa-chevron-right text-gray-400"></i>
                            </a>
                            
                            @if($payroll->status == 'draft')
                            <a href="{{ route('payrolls.edit', $payroll) }}" 
                               class="flex items-center justify-between p-3 rounded-lg border border-gray-200 hover:bg-gray-50 transition-colors duration-200">
                                <div class="flex items-center">
                                    <div class="w-10 h-10 rounded-lg bg-yellow-100 flex items-center justify-center mr-3">
                                        <i class="fas fa-edit text-yellow-600"></i>
                                    </div>
                                    <span class="font-medium">Edit Payroll</span>
                                </div>
                                <i class="fas fa-chevron-right text-gray-400"></i>
                            </a>
                            @endif
                            
                            <button onclick="window.print()" 
                                    class="w-full flex items-center justify-between p-3 rounded-lg border border-gray-200 hover:bg-gray-50 transition-colors duration-200">
                                <div class="flex items-center">
                                    <div class="w-10 h-10 rounded-lg bg-purple-100 flex items-center justify-center mr-3">
                                        <i class="fas fa-print text-purple-600"></i>
                                    </div>
                                    <span class="font-medium">Print/Export</span>
                                </div>
                                <i class="fas fa-chevron-right text-gray-400"></i>
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Print Styles -->
<style>
@media print {
    nav, .btn-primary, .btn-secondary, [x-data], [onclick] {
        display: none !important;
    }
    
    body {
        background: white !important;
    }
    
    .container {
        max-width: 100% !important;
        padding: 0 !important;
    }
    
    .rounded-xl, .shadow-sm, .border {
        border: 1px solid #ddd !important;
        box-shadow: none !important;
        border-radius: 0 !important;
    }
    
    .bg-white {
        background: white !important;
    }
    
    .text-gray-800 {
        color: #1f2937 !important;
    }
    
    .grid {
        display: block !important;
    }
    
    .lg\:col-span-2, .space-y-6 > * {
        margin-bottom: 20px !important;
        page-break-inside: avoid;
    }
}
</style>

<script>
// Function to format currency
function formatCurrency(amount) {
    return new Intl.NumberFormat('en-EG', {
        style: 'currency',
        currency: 'EGP',
        minimumFractionDigits: 2
    }).format(amount);
}

// Initialize tooltips
document.addEventListener('DOMContentLoaded', function() {
    // Add print functionality
    const printBtn = document.querySelector('[onclick="window.print()"]');
    if (printBtn) {
        printBtn.addEventListener('click', function() {
            window.print();
        });
    }
    
    // Add confirmation for status changes
    const statusForms = document.querySelectorAll('form[action*="process"], form[action*="mark-paid"]');
    statusForms.forEach(form => {
        form.addEventListener('submit', function(e) {
            const action = this.action.includes('process') ? 'process' : 'mark as paid';
            if (!confirm(`Are you sure you want to ${action} this payroll?`)) {
                e.preventDefault();
            }
        });
    });
});
</script>
@endsection